﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.AllowFileExtension;
/// <summary>
/// Controller For AllowFileExtensionController
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class AllowFileExtensionController(IMediator _mediator) : BaseController
{
    /// <summary>
    /// Creates the Allow file extension.
    /// </summary>
    /// <param name="command">The command.</param>
    /// <returns></returns>
    [HttpPost]
    [ClaimCheck("ADD_ALLOW_FILE_EXTENSIONS")]
    public async Task<IActionResult> CreateAllowFileExtension(AddAllowFileExtensionCommand command)
    {
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Updates the Allow file extension.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <param name="command">The update storage setting command.</param>
    /// <returns></returns>
    [HttpPut("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(WorkflowDto))]
    [ClaimCheck("EDIT_ALLOW_FILE_EXTENSIONS")]
    public async Task<IActionResult> UpdateAllowFileExtension(Guid id, UpdateAllowFileExtensionCommand command)
    {
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Allow file extension delete
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("{id}")]
    [ClaimCheck("DELETE_ALLOW_FILE_EXTENSIONS")]
    public async Task<IActionResult> DeleteAllowFileExtenstion(Guid id)
    {
        var query = new DeleteAllowFileExtensionCommand { Id = id };
        var result = await _mediator.Send(query);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the document status.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetAllowFileExtension(Guid id)
    {
        var query = new GetAllowFileExtensionQuery { Id = id };
        var result = await _mediator.Send(query);
        return Ok(result.Data);
    }

    /// <summary>
    /// Get all document status
    /// </summary>
    /// <returns></returns>
    [HttpGet(Name = "GetAllowFileExtension")]
    [Produces("application/json", "application/xml", Type = typeof(List<AllowFileExtensionDto>))]
    [AllowAnonymous]
    public async Task<IActionResult> GetAllowFileExtensions()
    {
        var query = new GetAllAllowFileExtensionQuery { };
        var result = await _mediator.Send(query);
        return Ok(result.Data);
    }
}
