﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.Audit;

/// <summary>
/// Controller for Audit
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class AuditController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Add Audit
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    public async Task<IActionResult> AddAudit(AddAuditCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Update Exist Audit By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <param name="updateAuditCommand"></param>
    /// <returns></returns>
    [HttpPut("{Id}")]
    [Produces("application/json", "application/xml", Type = typeof(AuditDto))]
    public async Task<IActionResult> UpdateAudit(Guid Id, UpdateAuditCommand updateAuditCommand)
    {
        updateAuditCommand.Id = Id;
        var result = await mediator.Send(updateAuditCommand);
        return GenerateResponse(result);

    }


    /// <summary>
    /// Get all Audit
    /// </summary>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(List<AuditDto>))]
    [ClaimCheck("VIEW_AUDIT")]
    public async Task<IActionResult> GetAllAudit([FromQuery] AuditResource auditResource)
    {
        var query = new GetAllAuditCommand
        {
            auditResource = auditResource
        };
        var result = await mediator.Send(query);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };

        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));

        return Ok(result);
    }
    /// <summary>
    /// Get Audit Reviews To Notify
    /// </summary>
    /// <param name="auditResource"></param>
    /// <returns></returns>
    [HttpGet("auditReview/notify")]
    [Produces("application/json", "application/xml", Type = typeof(List<AuditDto>))]
    public async Task<IActionResult> GetAuditReviewsToNotify([FromQuery] AuditResource auditResource)
    {
        var command = new GetAuditReviewersForNorificationCommand { };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Question Response By Question Id
    /// </summary>
    /// <param name="auditId"></param>
    /// <param name="questionId"></param>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{auditId}/question/{questionId}/response")]
    public async Task<IActionResult> GetQuestionResponseByQuestionId(Guid auditId, Guid questionId)
    {
        var command = new GetQuestionResponseByQuestionIdCommand
        {
            QuestionId = questionId,
            AuditId = auditId
        };
        var result = await mediator.Send(command);
        return Ok(result);
    }
    /// <summary>
    /// Get Question Response With Attachment By Question Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("question/{id}/response/attachment")]
    public async Task<IActionResult> GetQuestionResponseWithAttachmentByQuestionId(Guid id)
    {
        var command = new GetQuestionResponseByQuestionIdCommand { QuestionId = id };
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get AuditTemplate By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetAudit(Guid id)
    {
        var result = await mediator.Send(new GetAuditCommand() { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Audit Questions By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/questions")]
    public async Task<IActionResult> GetAuditQuestionsById(Guid id)
    {
        var result = await mediator.Send(new GetAuditCommand() { Id = id });
        return GenerateResponse(result);
    }
    /// <summary>
    /// Get Audit Reviwers By Audit Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/reviewers")]
    public async Task<IActionResult> GetAuditRevierersById(Guid id)
    {
        var result = await mediator.Send(new GetReviewersByAuditIdCommand() { AuditId = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Audit for dropdown
    /// </summary>
    /// <returns></returns>
    [HttpGet("dropdown")]
    [Produces("application/json", "application/xml", Type = typeof(List<IdNameDto>))]
    public async Task<IActionResult> GetAuditsForDropDown()
    {
        var getAuditListCommand = new GetAuditsForDropdownCommand
        {
        };
        var result = await mediator.Send(getAuditListCommand);
        return Ok(result);
    }

    /// <summary>
    /// Get audit Complince Rate
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpGet("AuditComplinceRate")]
    public async Task<IActionResult> GetAuditComplinceRate([FromQuery] GetAuditComplianceRateCommand command)
    {
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get Audit Status Overview
    /// </summary>
    /// <returns></returns>
    [HttpGet("status-overview")]
    [Produces("application/json", "application/xml", Type = typeof(List<NameCountDto>))]
    public async Task<IActionResult> GetAuditStatusOverview([FromQuery] GetAuditStatusOverviewCommand command)
    {
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get Audit By Department
    /// </summary>
    /// <returns></returns>
    [HttpGet("audit-by-department")]
    [Produces("application/json", "application/xml", Type = typeof(List<NameCountDto>))]
    public async Task<IActionResult> GetAuditByDepartment([FromQuery] GetAuditByDepartmentCommand command)
    {
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get Audit Report Details By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/report-details")]
    public async Task<IActionResult> GetAuditReportDetails(Guid id)
    {
        var result = await mediator.Send(new GetAuditReportDetailsCommand() { Id = id });
        return GenerateResponse(result);
    }



    /// <summary>
    /// Get all Audit For Review
    /// </summary>
    /// <returns></returns>
    [HttpGet("audit-for-review")]
    [Produces("application/json", "application/xml", Type = typeof(List<AuditDto>))]
    public async Task<IActionResult> GetAllAuditForReview([FromQuery] AuditResource auditResource)
    {
        var query = new GetAllAuditForReviewCommand
        {
            auditResource = auditResource
        };
        var result = await mediator.Send(query);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };

        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));

        return Ok(result);
    }
    /// <summary>
    /// Get Capas By Audit Id
    /// </summary>
    /// <param name="auditId"></param>
    /// <returns></returns>
    [HttpGet("capas/{auditId}")]
    public async Task<IActionResult> GetCapasByAuditId(Guid auditId)
    {
        var command = new GetCapasByAuditIdCommand { AuditId = auditId };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }
    /// <summary>
    /// Get Ncs By Audit Id
    /// </summary>
    /// <param name="auditId"></param>
    /// <returns></returns>
    [HttpGet("ncs/{auditId}")]
    public async Task<IActionResult> GetNcsByAuditId(Guid auditId)
    {
        var command = new GetNcsByAuditIdCommand { AuditId = auditId };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

}
