﻿using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;
using System;
using System.Threading.Tasks;

namespace QualityManagement.API.Controllers.Audit
{
    /// <summary>
    /// Controller for AuditAttachment
    /// </summary>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class AuditResponseAttachmentController(IMediator mediator) : BaseController
    {

        /// <summary>
        /// Save Audit Response Attachment
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> SaveAuditResponseAttachment(SaveAuditResponseAttachmentCommand command)
        {
            var result = await mediator.Send(command);
            return GenerateResponse(result);
        }


        /// <summary>
        /// Upload A Audit Response Attachment Chunk
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost("chunk")]
        public async Task<IActionResult> UploadAuditResponseAttachmentChunk([FromForm] UploadAuditResponseAttachmenttChunkCommand command)
        {
            var result = await mediator.Send(command);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Mark all chunks of a Attachment  as uploaded.
        /// </summary>
        /// <param name="markAttachmentChunksUploadedCommand"></param>
        /// <returns></returns>
        [HttpPost("chunk/uploadStatus")]
        [Produces("application/json", "application/xml", Type = typeof(AuditResponseAttachmentChunkStatus))]
        public async Task<IActionResult> MarkSessionMediaAllChunkUpload(MarkAuditResponseAttachmentChunksUploadedCommand markAttachmentChunksUploadedCommand)
        {
            var result = await mediator.Send(markAttachmentChunksUploadedCommand);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Get Download Audit Response Attachment chunk.
        /// </summary>
        /// <param name="auditResponseAttachmentId"></param>
        /// <param name="chunkIndex"></param>
        /// <returns></returns>
        [HttpGet("{auditResponseAttachmentId}/chunk/{chunkIndex}/download")]
        public async Task<IActionResult> DownloadAuditAttachment(Guid auditResponseAttachmentId, int chunkIndex)
        {
            var result = new DownloadAuditAttachmentCommand
            {
                AuditResponseAttachmentId = auditResponseAttachmentId,
                ChunkIndex = chunkIndex
            };

            var downloadAttachmentResponse = await mediator.Send(result);
            if (!downloadAttachmentResponse.Success)
            {
                return GenerateResponse(downloadAttachmentResponse);
            }

            return Ok(downloadAttachmentResponse.Data);
        }

        /// <summary>
        /// Delete Audit Response Attachment By Id
        /// </summary>
        /// <param name="Id"></param>
        /// <returns></returns>
        [HttpDelete("{Id}")]
        public async Task<IActionResult> DeleteAuditResponseAttachment(Guid Id)
        {
            var deleteAuditResponseAttachmentCommand = new DeleteAuditResponseAttachmentCommand { Id = Id };
            var result = await mediator.Send(deleteAuditResponseAttachmentCommand);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Gets the document token.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        [HttpGet("{id}/token")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentToken(Guid id)
        {
            var result = await mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
            return Ok(new { result = result.Data });
        }

        /// <summary>
        /// Get Document File By Token
        /// </summary>
        /// <param name="id"></param>
        /// <param name="officeViewerRequest"></param>
        /// <returns></returns>
        [HttpGet("{id}/officeviewer")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)
        {
            var command = new DownloadAuditResponseAttachmentChunkCommand
            {
                Id = id,
                Token = officeViewerRequest.Token
            };

            var downloadDocumentResponse = await mediator.Send(command);

            if (!downloadDocumentResponse.Success)
            {
                return GenerateResponse(downloadDocumentResponse);
            }

            var downloadDocument = downloadDocumentResponse.Data;
            return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
        }
    }
}
