﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.MediatR;
using QualityManagement.MediatR.capa;
using QualityManagement.MediatR.Capa;

namespace QualityManagement.API.Controllers.Capa;

/// <summary>
/// Controller for CapaAttachment
/// </summary>
[Route("api/[controller]")]
[ApiController]
public class CapaAttachmentController(IMediator _mediator) : BaseController
{
    /// <summary>
    /// Create A  CapaAttachment 
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    [ClaimCheck("ADD_CAPA_ATTACHMENT")]
    public async Task<IActionResult> AddCapaAttachment(AddCapaAttechmentCommand command)
    {
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// upload A CapaAttachment chunk
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost("Chunk")]
    public async Task<IActionResult> UploadCapaAttachmentChunk([FromForm] UploadCapaAttechmentChunkCommand command)
    {
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Mark all chunks of a Attachment  as uploaded.
    /// </summary>
    /// <param name="markAttachmentChunksUploadedCommand"></param>
    /// <returns></returns>
    [HttpPost("Chunk/uploadStatus")]
    [Produces("application/json", "application/xml", Type = typeof(CapaAttachmentChunkStatus))]
    public async Task<IActionResult> MarkCapaAttachmentAllChunkUpload(MarkCapaAttachmentChunksUploadedCommand markAttachmentChunksUploadedCommand)
    {
        var result = await _mediator.Send(markAttachmentChunksUploadedCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get all CapaAttachment by CapaRequest id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    [ClaimCheck("VIEW_CAPA_ATTACHMENT")]
    public async Task<IActionResult> GetAllCapaAttachmentByCapaRequestId(Guid id)
    {
        var result = await _mediator.Send(new GetAllCapaAttachmentByCapaRequestIdCommand() { CapaRequestId = id });
        return GenerateResponse(result);
    }



    /// <summary>
    /// Get Download CapaAttachment chunk.
    /// </summary>
    /// <param name="capaAttachmentId"></param>
    /// <param name="chunkIndex"></param>
    /// <returns></returns>
    [HttpGet("{capaAttachmentId}/chunk/{chunkIndex}/download")]
    public async Task<IActionResult> DownloadCapaAttachment(Guid capaAttachmentId, int chunkIndex)
    {
        var result = new DownloadCapaAttachmentCommand
        {
            CapaAttachmentId = capaAttachmentId,
            ChunkIndex = chunkIndex
        };

        var downloadAttachmentResponse = await _mediator.Send(result);
        if (!downloadAttachmentResponse.Success)
        {
            return GenerateResponse(downloadAttachmentResponse);
        }

        return Ok(downloadAttachmentResponse.Data);
    }

    /// <summary>
    /// Delete CapaAttachment By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    [ClaimCheck("DELETE_CAPA_ATTACHMENT")]
    public async Task<IActionResult> DeleteCapaAttachment(Guid Id)
    {
        var deleteCapaAttachmentCommand = new DeleteCapaAttachmentCommand { Id = Id };
        var result = await _mediator.Send(deleteCapaAttachmentCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the document token.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}/token")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentToken(Guid id)
    {
        var result = await _mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
        return Ok(new { result = result.Data });
    }

    /// <summary>
    /// Get Document File By Token
    /// </summary>
    /// <param name="id"></param>
    /// <param name="officeViewerRequest"></param>
    /// <returns></returns>
    [HttpGet("{id}/officeviewer")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)
    {
        var command = new DownloadCapaAttachmentChunkCommand
        {
            Id = id,
            Token = officeViewerRequest.Token
        };
        var downloadDocumentResponse = await _mediator.Send(command);
        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }
        var downloadDocument = downloadDocumentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }

}
