﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.Repository;

namespace QualityManagement.API.Controllers.Course;
/// <summary>
/// Controller for Course
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class CourseController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create A Course
    /// </summary>
    /// <param name="addCourseCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(CourseDto))]
    public async Task<IActionResult> CreateCourse(AddCourseCommand addCourseCommand)
    {
        var result = await mediator.Send(addCourseCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Update Course
    /// </summary>
    /// <param name="id"></param>
    /// <param name="updateCourseCommand"></param>
    /// <returns></returns>

    [HttpPut("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(CourseDto))]
    public async Task<IActionResult> UpdateCourse(Guid id, UpdateCourseCommand updateCourseCommand)
    {
        updateCourseCommand.Id = id;
        var result = await mediator.Send(updateCourseCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Course By id.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(CourseDto))]
    public async Task<IActionResult> GetCourse(Guid id)
    {
        var getCourseCommand = new GetCourseCommand { Id = id };
        var result = await mediator.Send(getCourseCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get All Courses
    /// </summary>
    /// <param name="courseResource"></param>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(CourseList))]
    public async Task<IActionResult> GetCourses([FromQuery] CourseResource courseResource)
    {
        var getCourseListCommand = new GetCourseListCommand
        {
            CourseResource = courseResource
        };
        var result = await mediator.Send(getCourseListCommand);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));
        return Ok(result);
    }

    /// <summary>
    /// Delete Course By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    public async Task<IActionResult> DeleteCourse(Guid Id)
    {
        var deleteCourseCommand = new DeleteCourseCommand { Id = Id };
        var result = await mediator.Send(deleteCourseCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Course for dropdown
    /// </summary>
    /// <param name="query"></param>
    /// <returns></returns>
    [HttpGet("search")]
    [Produces("application/json", "application/xml", Type = typeof(List<IdNameDto>))]
    public async Task<IActionResult> GetCoursesForDropDown([FromQuery] string query)
    {
        var getCompanyListCommand = new GetCourseForDropdownCommand
        {
            Query = query
        };
        var result = await mediator.Send(getCompanyListCommand);
        return Ok(result);
    }
}