﻿using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;
using QualityManagement.MediatR.CourseSessionMedias.DownloadSessionMedia;
using System;
using System.Text;
using System.Threading.Tasks;

namespace QualityManagement.API.Controllers.Course;
/// <summary>
/// Controller For CourseSessionMedia
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class CourseSessionMediaController(IMediator _mediator) : BaseController
{

    /// <summary>
    /// Get session media By session id.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(CourseDto))]
    public async Task<IActionResult> GetSessionMedia(Guid id)
    {
        var getCourseCommand = new GetSessionMediaCommand { CourseSessionId = id };
        var result = await _mediator.Send(getCourseCommand);
        return Ok(result);
    }

    /// <summary>
    /// Create a Session Media.
    /// </summary>
    /// <param name="addSessionMediaCommand"></param>
    /// <returns></returns>
    [DisableRequestSizeLimit]
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(CourseSessionMediaDto))]
    public async Task<IActionResult> AddSessionMedia([FromForm] AddSessionMediaCommand addSessionMediaCommand)
    {
        var result = await _mediator.Send(addSessionMediaCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Upload a Session Media chunk.
    /// </summary>
    /// <param name="uploadSessionMediaChunkCommand"></param>
    /// <returns></returns>
    [HttpPost("chunk/upload")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(CourseSessionMediaDto))]
    public async Task<IActionResult> UploadSessionMediaChunk([FromForm] UploadSessionMediaChunkCommand uploadSessionMediaChunkCommand)
    {
        var result = await _mediator.Send(uploadSessionMediaChunkCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Add a Session Media chunk.
    /// </summary>
    /// <param name="addSessionMediaChunkCommand"></param>
    /// <returns></returns>
    [HttpPost("chunk")]
    [Produces("application/json", "application/xml", Type = typeof(CourseSessionMediaDto))]
    public async Task<IActionResult> AddSessionMediaChunk(AddSessionMediaChunkCommand addSessionMediaChunkCommand)
    {
        var result = await _mediator.Send(addSessionMediaChunkCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Mark all chunks of a Session Media as uploaded.
    /// </summary>
    /// <param name="markMediaChunksUploadedCommand"></param>
    /// <returns></returns>
    [HttpPost("chunk/uploadStatus")]
    [Produces("application/json", "application/xml", Type = typeof(CourseSessionMediaDto))]
    public async Task<IActionResult> MarkSessionMediaAllChunkUpload(MarkMediaChunksUploadedCommand markMediaChunksUploadedCommand)
    {
        var result = await _mediator.Send(markMediaChunksUploadedCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Delete Course Media
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    public async Task<IActionResult> DeleteSessionMedia(Guid Id)
    {
        var deleteSessionMediaCommand = new DeleteSessionMediaCommand { Id = Id };
        var result = await _mediator.Send(deleteSessionMediaCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// get file chunk
    /// </summary>
    /// <param name="sessionMediaId"></param>
    /// <param name="chunkIndex"></param>
    /// <returns></returns>
    [HttpGet("{sessionMediaId}/chunk/{chunkIndex}/download")]
    public async Task<IActionResult> GetFileChunk(Guid sessionMediaId, int chunkIndex)
                {
        var command = new GetCourseSessionMediaChunkCommand
        {
            CourseSessionMediaId = sessionMediaId,
            ChunkIndex = chunkIndex
        };

        var documentChunkFile = await _mediator.Send(command);
        if (!documentChunkFile.Success)
        {
            return GenerateResponse(documentChunkFile);
        }

        return Ok(documentChunkFile.Data);
    }
    /// <summary>
    /// get Document chunk
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/chunks")]
    public async Task<IActionResult> DocumentChunks(Guid id)
    {
        var commnad = new GetSessionMediaChunksCommand
        {
            SessionMediaId = id,
        };

        var documentChunks = await _mediator.Send(commnad);
        if (!documentChunks.Success)
        {
            return StatusCode(documentChunks.StatusCode, documentChunks.Errors);
        }
        return Ok(documentChunks.Data);
    }
    /// <summary>
    /// Get Download Document
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/download")]
    public async Task<IActionResult> DownloadDocument(Guid id)
    {
        var commnad = new DownloadSessionMediaCommand
        {
            SessionMediaId = id,
        };

        var downloadDocumentResponse = await _mediator.Send(commnad);
        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;

        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }
    /// <summary>
    ///  Check Document Store AsChunk
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/isChunked")]
    [Produces("application/json", "application/xml", Type = typeof(bool))]
    [AllowAnonymous]
    public async Task<IActionResult> CheckDocumentStoreAsChunk(Guid id)
    {
        var checkDocumentStoreAsChunkCommand = new CheckMideaStoreAsChunkCommand
        {
            SessionMediaId = id
        };
        var result = await _mediator.Send(checkDocumentStoreAsChunkCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Read text Document
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/readText")]
    public async Task<IActionResult> ReadTextDocument(Guid id)
    {
        var commnad = new DownloadSessionMediaCommand
        {
            SessionMediaId = id
        };
        var downloadDocumentResponse = await _mediator.Send(commnad);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        string utfString = Encoding.UTF8.GetString(downloadDocument.Data, 0, downloadDocument.Data.Length);
        return Ok(new { result = new string[] { utfString } });
    }
    /// <summary>
    /// Get Document File By Token
    /// </summary>
    /// <param name="id"></param>
    /// <param name="token"></param>
    /// <returns></returns>
    [HttpGet("{id}/officeviewer")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] Guid token)
    {
        var getDocumentTokenCommand = new GetDocumentPathByTokenCommand
        {
            Id = id,
            Token = token,
        };

        var result = await _mediator.Send(getDocumentTokenCommand);
        if (!result)
        {
            return NotFound();
        }

        var commnad = new DownloadSessionMediaCommand
        {
            SessionMediaId = id,
        };

        var downloadDocumentResponse = await _mediator.Send(commnad);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }

    /// <summary>
    /// Gets the document token.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}/token")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentToken(Guid id)
    {
        var result = await _mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
        return Ok(new { result = result.Data });
    }


}
