﻿using System.Text.Json;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.Repository;

namespace QualityManagement.API.Controllers.DocumentAuditTrail;

/// <summary>
/// DocumentAuditTrail
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class DocumentAuditTrailController(IMediator _mediator) : ControllerBase
{
    /// <summary>
    /// Get All Document Audit Trail detail
    /// </summary>
    /// <param name="documentResource"></param>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(DocumentAuditTrailList))]
    [ClaimCheck("VIEW_DOCUMENT_AUDIT_TRAIL")]
    public async Task<IActionResult> GetDocumentAuditTrails([FromQuery] DocumentResource documentResource)
    {
        var getAllDocumentAuditTrailQuery = new GetAllDocumentAuditTrailQuery
        {
            DocumentResource = documentResource
        };
        var result = await _mediator.Send(getAllDocumentAuditTrailQuery);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));
        return Ok(result);
    }
    /// <summary>
    /// Add Document Audit Trail
    /// </summary>
    /// <param name="addDocumentAuditTrailCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(DocumentAuditTrailDto))]
    public async Task<IActionResult> AddDocumentAuditTrail(AddDocumentAuditTrailCommand addDocumentAuditTrailCommand)
    {
        var result = await _mediator.Send(addDocumentAuditTrailCommand);
        if (result.StatusCode != 200)
        {
            return StatusCode(result.StatusCode, result);
        }
        return StatusCode(StatusCodes.Status201Created, result);
    }
}
