﻿using System;
using System.Threading.Tasks;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.EmployeeCourseSession;
/// <summary>
/// Controller For EmployeeCourseSession
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class EmployeeCourseSessionController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create Employee CourseSession
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    public async Task<IActionResult> CreateEmployeeCourseSession(CreateEmployeeCourseSessionCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Employee course seesion
    /// </summary>
    /// <param name="link"></param>
    /// <returns></returns>
    [HttpGet("{link}")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseSessionDetailDto))]
    public async Task<IActionResult> GetEmployeeCourseSession(string link)
    {
        var command = new GetEmployeeCourseSessionByLinkCommand { Link = link };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Employee course seesion by id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/detail")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseSessionDetailDto))]
    public async Task<IActionResult> GetEmployeeCourseSessionById(Guid id)
    {
        var command = new GetEmployeeCourseSessionByIdCommand { Id = id };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Assigned Course Sessions By CourseId for current login user.
    /// </summary>
    /// <returns></returns>
    [HttpGet("{id}/sessions")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeAssignedCourseSessionDto))]
    public async Task<IActionResult> GetAssignedCourseSessionByCourseId(Guid id)
    {
        var command = new GetAssignedCourseSessionByCourseIdCommand()
        {
            CourseId = id
        };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Assigned Course Sessions By CourseId and EmployeeId
    /// </summary>
    /// <returns></returns>
    [HttpGet("{courseId}/sessions/{employeeId}")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeAssignedCourseSessionDto))]
    public async Task<IActionResult> GetAssignedCourseSessionByCourseId(Guid courseId, Guid employeeId)
    {
        var command = new GetEmployeeCourseSessionByCourseIdCommand()
        {
            CourseId = courseId,
            EmployeeId = employeeId
        };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Resume Last Session And Media by EmployeeCourse Link
    /// </summary>
    /// <param name="link"></param>
    /// <returns></returns>

    [HttpGet("session/media/{link}")]
    public async Task<IActionResult> GetEmployeeSessioMedia(string link)
    {
        var result = await mediator.Send(new GetResumeCourseSessionMedia() { Link = link });
        return GenerateResponse(result);
    }
     
    /// <summary>
    /// Get all media for a course session, with watched status (IsCompleted).
    /// </summary>
    /// <param name="link"></param>
    /// <param name="courseSessionId">ID of the course session</param>
    [HttpGet("status/{link}/{courseSessionId}")]
    public async Task<IActionResult> GetMediaStatusBySessionId(string link, Guid courseSessionId)
    {
        var result = await mediator.Send(new GetEmployeeCourseSessionMediaStatusCommand
        {
            CourseLink = link,
            CourseSessionId = courseSessionId
        });
        return GenerateResponse(result);
    }

}
