﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers;

/// <summary>
/// Controller for NonConformance
/// </summary>
/// <param name="mediator"></param>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class NonConformanceController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create A NonConformance
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(NonConformanceResponseDto))]
    [ClaimCheck("ADD_NON_CONFORMANCE")]
    public async Task<IActionResult> AddNonConformance(AddNonConformanceCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Update NonConformance by id
    /// </summary>
    /// <param name="id"></param>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPut("{id}")]
    [ClaimCheck("EDIT_NON_CONFORMANCE")]
    public async Task<IActionResult> UpdateNonConformance(Guid id, UpdateNonConformanceCommand command)
    {
        command.Id = id;
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Delete NonConformance by id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("{id}")]
    [ClaimCheck("DELETE_NON_CONFORMANCE")]
    public async Task<IActionResult> DeleteNonconformance(Guid id)
    {
        var result = await mediator.Send(new DeleteNonConformanceCommand() { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get All NonConformances
    /// </summary>
    /// <param name="nonConformanceResource"></param>
    /// <returns></returns>
    [HttpGet]
    [ClaimCheck("VIEW_NON_CONFORMANCE")]
    public async Task<IActionResult> GetNonConformances([FromQuery] NonConformanceResource nonConformanceResource)
    {
        var getNonConformanceListCommand = new GetNonConformanceListCommand()
        {
            NonConformanceResource = nonConformanceResource
        };
        var result = await mediator.Send(getNonConformanceListCommand);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));

        return Ok(result);
    }

    /// <summary>
    /// get NonConformanceResponse By NonConformance Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetNonConformanceResponse(Guid id)
    {
        var result = await mediator.Send(new GetNonconformanceResponseCommand() { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the new NcNumber.
    /// </summary>
    /// <returns></returns>
    [HttpGet("GetNewNcNumber")]
    public async Task<IActionResult> GetNewNcNumber()
    {
        var getNewNCNumberQuery = new GetNewNcNumberCommand();
        var response = await mediator.Send(getNewNCNumberQuery);
        return Ok(new
        {
            NcNumber = response
        });
    }

    /// <summary>
    /// Get All Audits
    /// </summary>
    /// <returns></returns>
    [HttpGet("audits")]
    public async Task<IActionResult> GetAllAudits()
    {
        var query = new GetAllNonConformanceAuditsCommand { };
        var result = await mediator.Send(query);
        return Ok(result);
    }

    /// <summary>
    /// Get A NonConformance Rate
    /// </summary>
    /// <returns></returns>
    [HttpGet("NcRate")]
    public async Task<IActionResult> GetNcRate()
    {
        var result = await mediator.Send(new GetNcRateCommand());
        return Ok(result);
    }

    /// <summary>
    /// Get A NonConformance Overview
    /// </summary>
    /// <returns></returns>
    [HttpGet("NonConformanceOverview")]
    [Produces("application/json", "application/xml", Type = typeof(List<NameCountDto>))]
    public async Task<IActionResult> GetNonConformanceOverview([FromQuery] GetNonConformanceOverviewCommand command)
    {
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get A NonConformance by severity
    /// </summary>
    /// <returns></returns>
    [HttpGet("NonConformanceBySeverity")]
    [Produces("application/json", "application/xml", Type = typeof(List<NameCountDto>))]
    public async Task<IActionResult> GetNonConformanceBySeverity([FromQuery] GetNonConformanceBySeverityCommand command)
    {
        var result = await mediator.Send(command);
        return Ok(result);
    }
}
