﻿using System;
using System.Collections.Generic;
using System.Text.Json;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.Role;

/// <summary>
/// Role
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class RoleController(IMediator _mediator, ILogger<RoleController> _logger) : ControllerBase
{
    /// <summary>
    /// Create A Role
    /// </summary>
    /// <param name="addRoleCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(RoleDto))]
    [ClaimCheck("CREATE_ROLE")]
    public async Task<IActionResult> AddRole(AddRoleCommand addRoleCommand)
    {
        var result = await _mediator.Send(addRoleCommand);
        if (result.StatusCode != 200)
        {
            _logger.LogError(result.StatusCode,
                            JsonSerializer.Serialize(result), "");
            return StatusCode(result.StatusCode, result);
        }
        return CreatedAtAction("GetRole", new { id = result.Id }, result);
    }
    /// <summary>
    /// Update Exist Role By Id
    /// </summary>
    /// <param name="id"></param>
    /// <param name="updateRoleCommand"></param>
    /// <returns></returns>

    [HttpPut("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(RoleDto))]
    [ClaimCheck("EDIT_ROLE")]
    public async Task<IActionResult> UpdateRole(Guid id, UpdateRoleCommand updateRoleCommand)
    {
        updateRoleCommand.Id = id;
        var result = await _mediator.Send(updateRoleCommand);
        if (result.StatusCode != 200)
        {
            _logger.LogError(result.StatusCode,
                            JsonSerializer.Serialize(result), "");
            return StatusCode(result.StatusCode, result);
        }
        return NoContent();
    }
    /// <summary>
    /// Get Role By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}", Name = "GetRole")]
    [Produces("application/json", "application/xml", Type = typeof(RoleDto))]
    public async Task<IActionResult> GetRole(Guid id)
    {
        _logger.LogTrace("GetRole");
        var getRoleQuery = new GetRoleQuery
        {
            Id = id
        };

        var result = await _mediator.Send(getRoleQuery);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Get All Roles
    /// </summary>
    /// <returns></returns>
    [HttpGet(Name = "GetRoles")]
    [Produces("application/json", "application/xml", Type = typeof(List<RoleDto>))]
    public async Task<IActionResult> GetRoles()
    {
        var getAllRoleQuery = new GetAllRoleQuery
        {
        };
        var result = await _mediator.Send(getAllRoleQuery);
        return Ok(result);
    }
    /// <summary>
    /// Delete Role By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    [Produces("application/json", "application/xml", Type = typeof(List<RoleDto>))]
    [ClaimCheck("DELETE_ROLE")]
    public async Task<IActionResult> DeleteRole(Guid Id)
    {
        var deleteOperationCommand = new DeleteRoleCommand
        {
            Id = Id
        };
        var result = await _mediator.Send(deleteOperationCommand);
        return StatusCode(result.StatusCode, result);
    }
}
