﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;

namespace QualityManagement.API.Controllers;
/// <summary>
/// Controller for SupplierCertification
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class SupplierCertificationController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create A SupplierCertification
    /// </summary>
    /// <param name="addSupplierCertificationCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(SupplierCertificationDto))]
    [ClaimCheck("ADD_SUPPLIER_CERTIFICATION")]
    public async Task<IActionResult> CreateSupplierCertification([FromForm] AddSupplierCertificationCommand addSupplierCertificationCommand)
    {
        var result = await mediator.Send(addSupplierCertificationCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get SupplierCertification By id.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(SupplierCertificationDto))]
    [ClaimCheck("VIEW_SUPPLIER_CERTIFICATION")]
    public async Task<IActionResult> GetSupplierCertification(Guid id)
    {
        var getSupplierCertificationCommand = new GetSupplierCertificationCommand { Id = id };
        var result = await mediator.Send(getSupplierCertificationCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Download Supplier Certification.
    /// </summary>
    /// <param name="supplierCertificationId"></param>
    /// <returns></returns>
    [HttpGet("{supplierCertificationId}/download")]
    public async Task<IActionResult> DownloadSupplierCertification(Guid supplierCertificationId)
    {
        var result = new DownloadSupplierCertificationCommand
        {
            SupplierCertificationId = supplierCertificationId,
        };

        var downloadAttachmentResponse = await mediator.Send(result);
        if (!downloadAttachmentResponse.Success)
        {
            return GenerateResponse(downloadAttachmentResponse);
        }
        var downloadDocument = downloadAttachmentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }

    /// <summary>
    /// Delete SupplierCertification By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    [ClaimCheck("DELETE_SUPPLIER_CERTIFICATION")]
    public async Task<IActionResult> DeleteSupplierCertification(Guid Id)
    {
        var deleteSupplierCertificationCommand = new DeleteSupplierCertificationCommand { Id = Id };
        var result = await mediator.Send(deleteSupplierCertificationCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the document token.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}/token")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentToken(Guid id)
    {
        var result = await mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
        return Ok(new { result = result.Data });
    }

    /// <summary>
    /// Get Document File By Token
    /// </summary>
    /// <param name="id"></param>
    /// <param name="officeViewerRequest"></param>
    /// <returns></returns>
    [HttpGet("{id}/officeviewer")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)

    {
        var command = new DownloadSupplierCertificationChunkCommand
        {
            Id = id,
            Token = officeViewerRequest.Token
        };

        var downloadDocumentResponse = await mediator.Send(command);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }
}