﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.UserOpenaiMsg;

/// <summary>
/// Controller For UserOpenAiMSG
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
public class UserOpenaiMsgController(IMediator _mediator) : BaseController
{
    /// <summary>
    /// Add User OpenAiMsg 
    /// </summary>
    /// <param name="command">The command.</param>
    /// <returns></returns>
    [HttpPost]
    public async Task<IActionResult> AddUserOpenaiMsg([FromBody] AddUserOpenaiMsgCommand command)
    {
        var result = await _mediator.Send(command);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return StatusCode(result.StatusCode, result.Data);
    }

    /// <summary>
    ///Get User OpenAIMsg By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetUserOpenaiMsgById(Guid id)
    {
        var query = new GetUserOpenaiMsgByIdQuery { Id = id };
        var result = await _mediator.Send(query);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return StatusCode(result.StatusCode, result.Data);
    }

    /// <summary>
    /// Get all User Open ai msg
    /// </summary>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(List<UserOpenaiMsgDto>))]
    [ClaimCheck("VIEW_AI_DOCUMENT_GENERATOR")]
    public async Task<IActionResult> GetAllUserOpenaiMsg([FromQuery] UserOpenaiMsgResource userOpenaiMsgResource)
    {
        var query = new GetAllUserOpenaiMsgCommand
        {
            userOpenaiMsgResource = userOpenaiMsgResource
        };
        var result = await _mediator.Send(query);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };

        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));

        return Ok(result);
    }

    /// <summary>
    /// Delete User OpenAiMsg
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("{id}")]
    [ClaimCheck("DELETE_AI_DOCUMENT_GENERATOR")]
    public async Task<IActionResult> DeleteUserOpenaiMsg(Guid id)
    {
        var query = new DeleteUserOpenaiMsgCommand { Id = id };
        var result = await _mediator.Send(query);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get User OpenAi Msg Response ById
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/response")]
    public async Task<IActionResult> GetUserOpenaiMsgResponseById(Guid id)
    {
        var query = new GetUserOpenaiMsgResponseByIdCommand { Id = id };
        var result = await _mediator.Send(query);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return GenerateResponse(result);
    }
}
