﻿using System;
using System.Threading.Tasks;
using Hangfire;
using MediatR;
using QualityManagement.MediatR;
using QualityManagement.MediatR.ReminderServices;
using QualityManagement.Repository;

namespace QualityManagement.API.Helpers;
/// <summary>
/// Jobservice is a class that manages the scheduling and execution of various jobs using Hangfire.
/// </summary>
public class JobService
{
    /// <summary>
    /// mediator is an instance of IMediator used to send commands and queries to the appropriate handlers.
    /// </summary>
    public IMediator _mediator { get; set; }
    private readonly IConnectionMappingRepository _connectionMappingRepository;

    /// <summary>
    /// constructor for JobService that initializes the mediator and connection mapping repository.
    /// </summary>
    /// <param name="mediator"></param>
    /// <param name="connectionMappingRepository"></param>
    public JobService(IMediator mediator, IConnectionMappingRepository connectionMappingRepository)
    {
        _mediator = mediator;
        _connectionMappingRepository = connectionMappingRepository;
    }
    /// <summary>
    /// method to start the scheduler and schedule various jobs using Hangfire.
    /// </summary>

    public void StartScheduler()
    {
        // * * * * *
        // 1 2 3 4 5

        // field #   meaning        allowed values
        // -------   ------------   --------------
        //    1      minute         0-59
        //    2      hour           0-23
        //    3      day of month   1-31
        //    4      month          1-12 (or use names)
        //    5      day of week    0-7 (0 or 7 is Sun, or use names)


        //Daily Reminder
        RecurringJob.AddOrUpdate("DailyReminder", () => DailyReminder(), Cron.Daily(0, 5), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Weekly Reminder
        RecurringJob.AddOrUpdate("WeeklyReminder", () => WeeklyReminder(), Cron.Daily(0, 10), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Monthy Reminder
        RecurringJob.AddOrUpdate("MonthlyReminder", () => MonthyReminder(), Cron.Daily(0, 20), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Quarterly Reminder
        RecurringJob.AddOrUpdate("QuarterlyReminder", () => QuarterlyReminder(), Cron.Daily(0, 30), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //HalfYearly Reminder
        RecurringJob.AddOrUpdate("HalfYearlyReminder", () => HalfYearlyReminder(), Cron.Daily(0, 40), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Yearly Reminder                                                                                
        RecurringJob.AddOrUpdate("YearlyReminder", () => YearlyReminder(), Cron.Daily(0, 50), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Customer Date
        RecurringJob.AddOrUpdate("CustomDateReminder", () => CustomDateReminderSchedule(), Cron.Daily(0, 59), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Reminder Scheduler To Send Email
        RecurringJob.AddOrUpdate("ReminderSchedule", () => ReminderSchedule(), "*/10 * * * *", new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 10 minutes

        //Send Email Scheduler To Send Email
        RecurringJob.AddOrUpdate("SendEmailScheduler", () => SendEmailScheduler(), "*/15 * * * *", new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 15 minutes

        //Reminder Scheduler To Send Email
        RecurringJob.AddOrUpdate("DocumentIndexing", () => DocumentIndexing(), "*/2 * * * *", new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 10 minutes

        //Send Email Reminder For Workflow Transition
        RecurringJob.AddOrUpdate("SendEmailReminderForWorkflowTransition", () => SendEmailReminderForWorkflowTransition(), "*/3 * * * *", new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 10 minutes

        //Permission Cleanup 
        RecurringJob.AddOrUpdate("PermissionCleanup", () => PermissionCleanupSchedule(), Cron.Daily(0, 59), new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 24 hours

        //Send Email Scheduler To Send Email  Asigned Cource Employee
        RecurringJob.AddOrUpdate("SendEmailSchedulerAsignCource", () => SendEmailSchedulerAsignCource(), "*/15 * * * *", new RecurringJobOptions { TimeZone = TimeZoneInfo.Local }); // Every 15 minutes

    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> DailyReminder()
    {
        return await _mediator.Send(new DailyReminderServicesQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> WeeklyReminder()
    {
        return await _mediator.Send(new WeeklyReminderServicesQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> MonthyReminder()
    {
        return await _mediator.Send(new MonthlyReminderServicesQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> QuarterlyReminder()
    {
        return await _mediator.Send(new QuarterlyReminderServiceQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> HalfYearlyReminder()
    {
        return await _mediator.Send(new HalfYearlyReminderServiceQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> YearlyReminder()
    {
        return await _mediator.Send(new YearlyReminderServicesQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> ReminderSchedule()
    {
        return await _mediator.Send(new ReminderSchedulerServiceQuery());
    }
    [Queue("reminder")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> CustomDateReminderSchedule()
    {
        return await _mediator.Send(new CustomDateReminderServicesQuery());
    }
    [Queue("cleanup")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> PermissionCleanupSchedule()
    {
        return await _mediator.Send(new CleanupExpiredPermissionsCommand());
    }
    [Queue("emailnotification")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> SendEmailScheduler()
    {

        return await _mediator.Send(new SendEmailSchedulerCommand());

    }
    [Queue("workflow")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> SendEmailReminderForWorkflowTransition()
    {
        return await _mediator.Send(new SendEmailReminderForWorkflowTransitionCommand());
    }
    [Queue("documentindex")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> DocumentIndexing()
    {

        return await _mediator.Send(new AddDocumentIndexContentCommand());

    }
    [Queue("archievedocument")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> ArchiveRetentionDocumentToDelete()
    {
        return await _mediator.Send(new ArchiveRetentionDocumentBackgroundServiceCommand());
    }
    [Queue("archievedocument")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public async Task<bool> ArchieveOrDeleteDocummentBaseOnConfigurationOnDocumentOnAction()
    {
        return await _mediator.Send(new RunArchiveRetensionOnDocumentActionConfigurationCommand());
    }

    /// <summary>
    /// Send email scheduler method to send emails Asign Cource to Employee.
    /// </summary>
    /// <returns></returns>
    [Queue("emailnotification")]
    [AutomaticRetry(Attempts = 3, DelaysInSeconds = new[] { 60, 300, 900 })]
    [DisableConcurrentExecution(3600)]
    public bool SendEmailSchedulerAsignCource()
    {
        var schedulerStatus = _connectionMappingRepository.GetEmailSchedulerStatus();
        if (!schedulerStatus)
        {
            _connectionMappingRepository.SetEmailSchedulerStatus(true);
            var result = _mediator.Send(new SendEmailSchedulerAssignCourseCommand()).GetAwaiter().GetResult();
            _connectionMappingRepository.SetEmailSchedulerStatus(false);
            return result;
        }
        return true;
    }
}
