﻿using Microsoft.AspNetCore.Http;
using Microsoft.IdentityModel.Tokens;
using QualityManagement.Data;
using System;
using System.IdentityModel.Tokens.Jwt;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace QualityManagement.API.Helpers;

/// <summary>
/// JwtMiddleware is a custom middleware that handles JWT authentication in the ASP.NET Core application.
/// </summary>
public class JwtMiddleware
{
    private readonly RequestDelegate _next;
    private readonly JwtSettings _settings = null;
    /// <summary>
    /// JwtMiddleware constructor for initializing the middleware with the next request delegate and JWT settings.
    /// </summary>
    /// <param name="next"></param>
    /// <param name="settings"></param>
    public JwtMiddleware(
        RequestDelegate next,
         JwtSettings settings)
    {
        _next = next;
        _settings = settings;
    }
    /// <summary>
    /// Invoke method is called to handle the incoming HTTP request. It extracts the JWT token from the request headers and validates it.
    /// </summary>
    /// <param name="context"></param>
    /// <returns></returns>
    public async Task Invoke(HttpContext context)
    {
        var token = context.Request.Headers["Authorization"].FirstOrDefault()?.Split(" ").Last();

        if (token != null)
            attachUserToContext(context, token);

        await _next(context);
    }
    /// <summary>
    /// Method to attach the user to the context if the JWT token is valid. It validates the token and extracts the user ID from it.
    /// </summary>
    /// <param name="context"></param>
    /// <param name="token"></param>
    private void attachUserToContext(HttpContext context, string token)
    {
        try
        {
            var tokenHandler = new JwtSecurityTokenHandler();
            var key = Encoding.ASCII.GetBytes(_settings.Key);
            tokenHandler.ValidateToken(token, new TokenValidationParameters
            {
                ValidateIssuerSigningKey = true,
                IssuerSigningKey = new SymmetricSecurityKey(key),
                ValidateIssuer = false,
                ValidateAudience = false,
                // set clockskew to zero so tokens expire exactly at token expiration time (instead of 5 minutes later)
                ClockSkew = TimeSpan.Zero
            }, out SecurityToken validatedToken);
            var jwtToken = (JwtSecurityToken)validatedToken;
            var userId = jwtToken.Claims.First(x => x.Type == "sub").Value;
            // attach user to context on successful jwt validation
            //context.Items["User"] = userId;
        }
        catch
        {
            // do nothing if jwt validation fails
            // user is not attached to context so request won't have access to secure routes
        }
    }
}
