﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddActionCommandHandler(
    IActionRepository _actionRepository,
    IPageRepository _pageRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper _mapper,
    ILogger<AddActionCommandHandler> _logger) : IRequestHandler<AddActionCommand, ServiceResponse<ActionDto>>
{
    public async Task<ServiceResponse<ActionDto>> Handle(AddActionCommand request, CancellationToken cancellationToken)
    {
        var entityExist = await _actionRepository.FindBy(c => c.PageId == request.PageId && c.Name.Trim().ToLower() == request.Name.Trim().ToLower()).FirstOrDefaultAsync();
        if (entityExist != null)
        {
            _logger.LogError("Action already exist.");
            return ServiceResponse<ActionDto>.Return409("Action already exist.");
        }

        var page = await _pageRepository.FindAsync(request.PageId);
        if (page == null)
        {
            _logger.LogError("Page does not exists.");
            return ServiceResponse<ActionDto>.Return404("Page does not exists.");
        }

        var entity = _mapper.Map<Data.Action>(request);
        entity.Id = Guid.NewGuid();
        _actionRepository.Add(entity);

        if (await _uow.SaveAsync() <= 0)
        {
            return ServiceResponse<ActionDto>.Return500();
        }

        var entityDto = _mapper.Map<ActionDto>(entity);
        return ServiceResponse<ActionDto>.ReturnResultWith200(entityDto);
    }
}
