﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class GetAuditByDepartmentCommandHandler(
    IAuditRepository auditRepository,
    ILogger<GetAuditStatusOverviewCommandHandler> logger) : IRequestHandler<GetAuditByDepartmentCommand, List<NameCountDto>>
{
    public async Task<List<NameCountDto>> Handle(GetAuditByDepartmentCommand request, CancellationToken cancellationToken)
    {
        try
        {
            IQueryable<AuditTemplate> auditTemplates;

            if (request.FromDate.HasValue && request.ToDate.HasValue)
            {
                var startDate = request.FromDate.Value;
                var endDate = request.ToDate.Value;

                DateTime minDate = new DateTime(startDate.Year, startDate.Month, startDate.Day, 0, 0, 0);
                DateTime maxDate = new DateTime(endDate.Year, endDate.Month, endDate.Day, 23, 59, 59);

                auditTemplates = auditRepository.All
                    .Include(a => a.AuditTemplate)
                    .Where(c => c.CreatedDate >= minDate && c.CreatedDate <= maxDate)
                    .Select(a => a.AuditTemplate);
            }
            else
            {
                auditTemplates = auditRepository.All
                    .Include(a => a.AuditTemplate)
                    .Select(a => a.AuditTemplate);
            }

            var departmentAuditOverview = await auditTemplates
               .GroupBy(a => a.Department.Name)
               .Select(g => new NameCountDto
               {
                   Name = g.Key,
                   Count = g.Count()
               })
               .ToListAsync(cancellationToken);

            return departmentAuditOverview;
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while handling GetAuditByDepartmentCommand");
            return new List<NameCountDto>();
        }
    }
}
