﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class GetAuditCommandHandler(
    IAuditRepository _auditRepository,
    IMapper _mapper,
    ILogger<GetAuditCommandHandler> _logger) : IRequestHandler<GetAuditCommand, ServiceResponse<AuditDto>>
{
    public async Task<ServiceResponse<AuditDto>> Handle(GetAuditCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await _auditRepository.All
                .Include(c => c.AuditTemplate)
                    .ThenInclude(a => a.Questions)
                        .ThenInclude(c => c.QuestionOptions)
                .Include(c => c.AuditTemplate)
                    .ThenInclude(c => c.Department)
                .Include(c => c.Auditor)
                .Include(c => c.NonConformances)
                    .ThenInclude(c => c.NonConformanceResponses)
                .Include(c => c.CapaRequests)
                .Include(c => c.AuditReviewers.OrderByDescending(c => c.ReviewedOn))
                    .ThenInclude(c => c.Reviewer)
                 .Include(c => c.AuditReviewers.OrderByDescending(c => c.ReviewedOn))
                    .ThenInclude(c => c.Submitter)
                .Where(a => a.Id == request.Id)
                  .AsNoTracking()
                .AsSplitQuery()
                .FirstOrDefaultAsync();

            if (existEntity == null)
            {
                return ServiceResponse<AuditDto>.Return404();
            }

            var entityDto = _mapper.Map<AuditDto>(existEntity);
            return ServiceResponse<AuditDto>.ReturnResultWith200(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while getting Audit");
            return ServiceResponse<AuditDto>.Return500("Error while getting Audit");
        }
    }
}

