﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class AddAuditResponseCommandHandler(
    IAuditResponseRepository auditResponseRepository,
    IAuditResponseLogRepository auditResponseLogRepository,
    IAuditTemplateQuestionRepository auditTemplateQuestionRepository,
      IAuditRepository _auditRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   UserInfoToken userInfoToken,
   ILogger<AddAuditCommandHandler> _logger
    ) : IRequestHandler<AddAuditResponseCommand, ServiceResponse<AuditResponseDto>>
{
    public async Task<ServiceResponse<AuditResponseDto>> Handle(AddAuditResponseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await auditResponseRepository.All
                .Where(c => c.AuditId == request.AuditId && c.QuestionId == request.QuestionId)
                .FirstOrDefaultAsync();

            if (existEntity != null)
            {
                return ServiceResponse<AuditResponseDto>.Return409("Audit Response already exist.");
            }
            var entity = _mapper.Map<AuditResponse>(request);
            var audit = await _auditRepository.All
                .FirstOrDefaultAsync(a => a.Id == request.AuditId);
            int totalScore = await auditResponseRepository.All
                 .Where(c => c.AuditId == request.AuditId)
                 .SumAsync(c => c.Score);
            totalScore += request.Score;
            var maxscore = await auditTemplateQuestionRepository
                .All
                .Where(c => c.AuditTemplateId == audit.AuditTemplateId).SumAsync(c => c.MaxScore);

            if (audit != null && maxscore > 0)
            {
                audit.EffectiveScore = totalScore;
                _auditRepository.Update(audit);
            }
            auditResponseRepository.Add(entity);
            var auditResponseLog = _mapper.Map<AuditResponseLog>(entity);
            auditResponseLog.Id = Guid.NewGuid();
            auditResponseLog.Status = AUDIT_STATUS.INPROGRESS;
            auditResponseLog.AuditId = entity.AuditId;
            auditResponseLog.CreatedBy = userInfoToken.Id;
            auditResponseLog.CreatedDate = DateTime.UtcNow;
            auditResponseLog.LogStatus = LogStatus.Updated;
            var question = auditTemplateQuestionRepository.AllIncluding(c => c.QuestionOptions).Where(x => x.Id == entity.QuestionId).FirstOrDefault();
            if (question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.SINGLE_SELECT || question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.MULTI_SELECT)
            {
                var responseIds = entity.Response.Split(',')
                    .Select(id => Guid.Parse(id.Trim()))
                    .ToList();
                var selectedOptions = question.QuestionOptions
                    .Where(c => responseIds.Contains(c.Id))
                    .Select(c => c.OptionText);

                auditResponseLog.Response = string.Join(", ", selectedOptions);
            }
            auditResponseLogRepository.Add(auditResponseLog);
            if (await _uow.SaveAsync() < 0)
            {
                return ServiceResponse<AuditResponseDto>.Return500();
            }

            var entityDto = _mapper.Map<AuditResponseDto>(entity);
            return ServiceResponse<AuditResponseDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving Audit Response");
            return ServiceResponse<AuditResponseDto>.Return500("Error while Saving Audit Response");
        }
    }
}
