﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class DownloadAuditAttachmentCommandHandler(
        IAuditResponseAttachmentRepository auditResponseAttachmentRepository,
         IAuditResponseAttachmentChunkRepository auditResponseAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadAuditAttachmentCommand, ServiceResponse<AuditAttachmentDownload>>
{

    public async Task<ServiceResponse<AuditAttachmentDownload>> Handle(DownloadAuditAttachmentCommand request, CancellationToken cancellationToken)
    {
        var auditAttachment = auditResponseAttachmentRepository.Find(request.AuditResponseAttachmentId);
        if (auditAttachment == null)
        {
            return ServiceResponse<AuditAttachmentDownload>.ReturnFailed(404, "AuditAttachment is not found");
        }

        var auditAttachmentChunk = await auditResponseAttachmentChunkRepository.All
            .Where(c => c.AuditResponseAttachmentId == request.AuditResponseAttachmentId && c.ChunkIndex == request.ChunkIndex)
            .FirstOrDefaultAsync();

        if (auditAttachmentChunk == null)
        {
            return ServiceResponse<AuditAttachmentDownload>.ReturnFailed(404, "AuditAttachment Chunk is not found");
        }

        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(auditAttachment.StorageSettingId);

        if (storeageSetting == null)
        {
            return ServiceResponse<AuditAttachmentDownload>.ReturnFailed(404, "Storage setting not found");
        }
        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
        var fileResult = await storageService.DownloadFileAsync(auditAttachmentChunk.Url, storeageSetting.JsonValue, auditAttachment.Key, auditAttachment.IV);
        if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
        {
            var extension = auditAttachmentChunk.Extension;
            if (!extension.StartsWith("."))
                extension = "." + extension;
            AuditAttachmentDownload auditAttachmentDownload = new AuditAttachmentDownload
            {
                Data = Convert.ToBase64String(fileResult.FileBytes),
                ContentType = FileHelper.GetMimeTypeByExtension(extension),
                FileName = auditAttachmentChunk.Url,
                ChunkIndex = auditAttachmentChunk.ChunkIndex
            };
            return ServiceResponse<AuditAttachmentDownload>.ReturnResultWith200(auditAttachmentDownload);
        }
        return ServiceResponse<AuditAttachmentDownload>.ReturnFailed(500, fileResult.ErrorMessage);
    }
}
