﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class UpdateAuditResponseCommandHandler(
   IAuditResponseRepository auditResponseRepository,
   UserInfoToken userInfoToken,
   IAuditTemplateQuestionRepository auditTemplateQuestionRepository,
   IAuditResponseLogRepository auditResponseLogRepository,
     IAuditRepository _auditRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddAuditCommandHandler> _logger) : IRequestHandler<UpdateAuditResponseCommand, ServiceResponse<AuditResponseDto>>
{
    public async Task<ServiceResponse<AuditResponseDto>> Handle(UpdateAuditResponseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await auditResponseRepository.All
            .Where(c => c.AuditId == request.AuditId && c.QuestionId == request.QuestionId)
            .FirstOrDefaultAsync();
            if (existEntity == null)
            {
                return ServiceResponse<AuditResponseDto>.Return409("Audit Response Not Found");
            }
            var entity = _mapper.Map(request, existEntity);
            int totalScore = await auditResponseRepository.All
                 .Where(c => c.AuditId == request.AuditId && c.Id != request.Id)
                 .SumAsync(c => c.Score);
            totalScore += request.Score;

            var audit = await _auditRepository.All
                .FirstOrDefaultAsync(a => a.Id == request.AuditId);

            var maxscore = await auditTemplateQuestionRepository
                .All
                .Where(c => c.AuditTemplateId == audit.AuditTemplateId).SumAsync(c => c.MaxScore);

            if (audit != null && maxscore > 0)
            {
                audit.EffectiveScore = totalScore;
                _auditRepository.Update(audit);
            }
            auditResponseRepository.Update(entity);
            var auditResponseLog = _mapper.Map<AuditResponseLog>(entity);
            auditResponseLog.Id = Guid.NewGuid();
            auditResponseLog.Status = AUDIT_STATUS.INPROGRESS;
            auditResponseLog.AuditId = entity.AuditId;
            auditResponseLog.CreatedBy = userInfoToken.Id;
            auditResponseLog.CreatedDate = DateTime.UtcNow;
            auditResponseLog.LogStatus = LogStatus.Updated;
            var question = auditTemplateQuestionRepository.AllIncluding(c => c.QuestionOptions).Where(x => x.Id == entity.QuestionId).FirstOrDefault();
            if (question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.SINGLE_SELECT || question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.MULTI_SELECT)
            {
                var responseIds = entity.Response.Split(',')
                    .Select(id => Guid.Parse(id.Trim()))
                    .ToList();
                var selectedOptions = question.QuestionOptions
                    .Where(c => responseIds.Contains(c.Id))
                    .Select(c => c.OptionText);

                auditResponseLog.Response = string.Join(", ", selectedOptions);
            }
            auditResponseLogRepository.Add(auditResponseLog);

            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditResponseDto>.Return500();
            }
            var entityDto = _mapper.Map<AuditResponseDto>(entity);
            return ServiceResponse<AuditResponseDto>.ReturnResultWith200(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while updating Audit Response");
            return ServiceResponse<AuditResponseDto>.Return500("Error while updating Audit Response");
        }
    }
}
