﻿using System;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class UploadAuditResponseAttachmenttChunkCommandHandler(
        IAuditResponseAttachmentChunkRepository auditResponseAttachmentChunkRepository,
        IAuditResponseAttachmentRepository auditResponseAttachmentRepository,
        IStorageSettingRepository _storageSettingRepository,
       StorageServiceFactory _storeageServiceFactory,
       IUnitOfWork<QMSDbContext> _uow,
       IMapper _mapper,
       ILogger<UploadNonConformanceRespAttachmenttChunkCommandHandler> logger
   ) : IRequestHandler<UploadAuditResponseAttachmenttChunkCommand, ServiceResponse<AuditResponseAttachmentChunkDto>>
{
    public async Task<ServiceResponse<AuditResponseAttachmentChunkDto>> Handle(UploadAuditResponseAttachmenttChunkCommand request, CancellationToken cancellationToken)
    {
        try
        {
            if (request.File == null)
            {
                return ServiceResponse<AuditResponseAttachmentChunkDto>.ReturnFailed(409, "Please select the file.");
            }

            var ncAttach = auditResponseAttachmentRepository.Find(request.AuditResponseAttachmentId);
            if (ncAttach == null)
            {
                return ServiceResponse<AuditResponseAttachmentChunkDto>.ReturnFailed(404, "Document not found.");
            }

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(ncAttach.StorageSettingId);

            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

            var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, ncAttach.Extension, ncAttach.Key, ncAttach.IV);

            if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
            {
                return ServiceResponse<AuditResponseAttachmentChunkDto>.Return422("Settings are not properly setup.");
            }

            var entity = new AuditResponseAttachmentChunk()
            {
                Id = Guid.NewGuid(),
                ChunkIndex = request.ChunkIndex,
                AuditResponseAttachmentId = request.AuditResponseAttachmentId,
                Extension = request.Extension,
                Size = request.Size,
                Url = fileNameKeyValut.FileName
            };

            auditResponseAttachmentChunkRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditResponseAttachmentChunkDto>.ReturnFailed(500, "Error While Added Document");
            }

            var entityDto = _mapper.Map<AuditResponseAttachmentChunkDto>(entity);
            return ServiceResponse<AuditResponseAttachmentChunkDto>.ReturnResultWith200(entityDto);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while storing AuditResponseAttachmentChunk.");
            return ServiceResponse<AuditResponseAttachmentChunkDto>.Return500("Error while storing AuditResponseAttachmentChunk");
        }

    }
}
