﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class AddAuditReviewerCommandHandler(IAuditReviewerRepository auditReviewerRepository,
   IMapper _mapper,
   IAuditRepository auditRepository,
   UserInfoToken userInfoToken,
   IAuditResponseLogRepository auditResponseLogRepository,
   IUserNotificationRepository userNotificationRepository,
   ISendEmailRepository sendEmailRepository,
   IUserRepository userRepository,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddAuditReviewerCommandHandler> _logger) : IRequestHandler<AddAuditReviewerCommand, ServiceResponse<AuditReviewerDto>>
{
    public async Task<ServiceResponse<AuditReviewerDto>> Handle(AddAuditReviewerCommand request, CancellationToken cancellationToken)
    {
        try
        {

            var entity = _mapper.Map<AuditReviewer>(request);
            var audit = await auditRepository.FindAsync(request.AuditId);
            if (audit != null)
            {
                audit.Status = AUDIT_STATUS.INREVIEW;
                auditRepository.Update(audit);
                var auditResponseLog = _mapper.Map<AuditResponseLog>(audit);
                auditResponseLog.Id = Guid.NewGuid();
                auditResponseLog.CreatedBy = userInfoToken.Id;
                auditResponseLog.CreatedDate = DateTime.UtcNow;
                auditResponseLog.LogStatus = LogStatus.Updated;
                auditResponseLogRepository.Add(auditResponseLog);
            }
            entity.ReviewedOn = DateTime.UtcNow;
            entity.AuditStatus = AUDIT_STATUS.INREVIEW;
            entity.SubmitterId = userInfoToken.Id;
            auditReviewerRepository.Add(entity);
            var notification = new NotificationAuditDto
            {
                AuditId = audit.Id,
                UserId = userInfoToken.Id,
                Message = $"You have been assigned the audit \"{audit.Title}\" for review.",
                Title = audit.Title,
                NotificationsType = NotificationsType.AUDITINREVIEW
            };
            userNotificationRepository.AddUserNotificationByAudit(notification);
            var reviewerUserInfo = await userRepository.FindAsync(request.ReviewerId);
            var currentUserInfo = await userRepository.FindAsync(userInfoToken.Id);
            sendEmailRepository.AddAuditEmails(new SendEmail
            {
                Email = reviewerUserInfo.Email,
                FromEmail = currentUserInfo.Email,
                FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                ToName = reviewerUserInfo.FirstName + ' ' + reviewerUserInfo.LastName,
                CreatedBy = userInfoToken.Id,
                CreatedDate = DateTime.UtcNow,
            }, $"You have been assigned the audit \"{audit.Title}\" for review.");
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditReviewerDto>.Return500();
            }
            var entityDto = _mapper.Map<AuditReviewerDto>(entity);
            return ServiceResponse<AuditReviewerDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving AuditReviwer");
            return ServiceResponse<AuditReviewerDto>.Return500("Error while Saving AuditReviwer");
        }
    }
}
