﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Linq.Dynamic.Core;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class AddCapaActionCommandHandler(
    ICapaActionRepository _capaActionRepository,
    ICapaRequestRepository _capaRequestRepository,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddCapaActionCommandHandler> _logger,
    IUserRepository _userRepository,
    IEmailSMTPSettingRepository _emailSMTPSettingRepository,
    ISendEmailRepository _sendEmailRepository,
    UserInfoToken _userInfoToken,
    IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddCapaActionCommand, ServiceResponse<CapaActionDto>>
{
    public async Task<ServiceResponse<CapaActionDto>> Handle(AddCapaActionCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existCapaRequset = await _capaRequestRepository.All
                .Where(c => c.Id == request.CapaRequestId).FirstOrDefaultAsync();
            if (existCapaRequset == null)
            {
                return ServiceResponse<CapaActionDto>.Return404("CapaRootCause not found");
            }
            var entity = _mapper.Map<CapaAction>(request);

            if(request.Status == CapaActionStatus.COMPLETED)
            {
                entity.CompletedDate = DateTime.UtcNow;
            }
            _capaActionRepository.Add(entity);

            var currentUserId = _userInfoToken.Id;
            var user = await _userRepository.All.Where(c => c.Id == currentUserId).FirstOrDefaultAsync();
            var asignedUser = await _userRepository.All.Where(c => c.Id == entity.AssignToId).FirstOrDefaultAsync();
            //var defaultSmtp = await _emailSMTPSettingRepository.FindBy(c => c.IsDefault).FirstOrDefaultAsync();
            try
            {
                if (asignedUser != null)
                {
                    var emailmessage = $"You have been assigned a new Corrective and Preventive Action (CAPA) on : <b>{existCapaRequset.Title}</b>";
                    var subject = "Corrective and Preventive Action Assignment";
                    var mainTitle = "CAPA Action Assignment Notification";
                    _sendEmailRepository.AddAsignmentEmails(new SendEmail
                    {
                        Email = asignedUser.Email,
                        FromName = user.FirstName + ' ' + user.LastName,
                        ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                        CreatedBy = user.Id,
                        CreatedDate = DateTime.UtcNow,
                    }, emailmessage, mainTitle, subject);
                }
            } 
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "email does not sent");
            }
            var notificationDto=new UserNotificationDto
            {
                Id = entity.Id,
                UserId = entity.AssignToId.Value,
                NotificationsType = NotificationsType.CAPA_ACTION,
                Title = $"{entity.ActionType}"
            };
            _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
            if (await _uow.SaveAsync(cancellationToken) < 0) 
            {
                return ServiceResponse<CapaActionDto>.Return500();
            }

            if (user != null)
            {
                entity.CreatedByUser = user;
            }
            if (asignedUser != null)
            {
                entity.AssignTo = asignedUser;
            }
            var entityDto = _mapper.Map<CapaActionDto>(entity);

            return ServiceResponse<CapaActionDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving CapaAction");
            return ServiceResponse<CapaActionDto>.Return500("Error while saving CapaAction");
        }
    }
}
