﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR.capa;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class MarkCapaAttachmentChunksUploadedCommandhandler(
        ICapaAttachmentRepository _capaAttachmentRepository,
        ICapaAttachmentChunkRepository _capaAttachmentChunkRepository,
    IStorageSettingRepository _storageSettingRepository,
    StorageServiceFactory _storageServiceFactory,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<MarkCapaAttachmentChunksUploadedCommandhandler> _logger
        ) : IRequestHandler<MarkCapaAttachmentChunksUploadedCommand, ServiceResponse<CapaAttachmentChunkStatus>>
    {
        public async Task<ServiceResponse<CapaAttachmentChunkStatus>> Handle(MarkCapaAttachmentChunksUploadedCommand request, CancellationToken cancellationToken)
        {

            try
            {
                var capaAttachment = await _capaAttachmentRepository.All
                    .Where(c => c.Id == request.CapaAttachmentId).FirstOrDefaultAsync();

                if (capaAttachment == null)
                {
                    return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(404, "capaAttachment not found.");
                }

                var capaChunks = _capaAttachmentChunkRepository
                    .All
                    .Where(c => c.CapaAttachmentId == capaAttachment.Id)
                    .OrderByDescending(c => c.ChunkIndex)
                    .ToList();

                if (capaChunks.Count == 0)
                {
                    return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(404, " capaAttachment Chunks not found.");
                }

                if (capaAttachment.TotalChunk == capaChunks.Count())
                {
                    capaAttachment.IsAllChunkUploaded = request.Status;

                    _capaAttachmentRepository.Update(capaAttachment);

                    if (await _uow.SaveAsync(cancellationToken) < 0)
                    {
                        return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding capaAttachment");
                    }
                    var response = new CapaAttachmentChunkStatus { CapaAttachmentId = capaAttachment.Id, Status = request.Status };
                    return ServiceResponse<CapaAttachmentChunkStatus>.ReturnResultWith200(response);
                }
                else
                {
                    if (capaChunks.Count() > 0)
                    {
                        foreach (var documentChunk in capaChunks)
                        {
                            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(capaAttachment.StorageSettingId);
                            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                            await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                        }
                        _capaAttachmentChunkRepository.RemoveRange(capaChunks);
                    }

                    _capaAttachmentRepository.Remove(capaAttachment);
                    if (await _uow.SaveAsync() <= 0)
                    {
                        return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding capaAttachment");
                    }

                    return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "Error while marking capaAttachment chunks uploaded.");
                return ServiceResponse<CapaAttachmentChunkStatus>.ReturnFailed(500, "Error while marking capaAttachment chunks uploaded.");
            }
        }
    }
}
