﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class AddCapaRequestCommandHandler(
    ICapaRequestRepository _capaRequestRepository,
    ICapaRequestLogRepository _capaRequestLogRepository,
    UserInfoToken userInfoToken,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddCapaRequestCommandHandler> _logger) : IRequestHandler<AddCapaRequestCommand, ServiceResponse<CapaRequestDto>>
{
    public async Task<ServiceResponse<CapaRequestDto>> Handle(AddCapaRequestCommand request, CancellationToken cancellationToken)

    {
        try
        {
            var entityExist = await _capaRequestRepository.All
                .Where(c => c.Title.ToLower() == request.Title.ToLower())
                .FirstOrDefaultAsync();
            if (entityExist != null)
            {
                return ServiceResponse<CapaRequestDto>.Return409("CapaRequest title already exist");
            }
            var entity = _mapper.Map<CapaRequest>(request);
            entity.Id = Guid.NewGuid();
            var newRefNumber = await _capaRequestRepository.GenerateCapaNumberAsync();
            entity.ReferenceNo = newRefNumber;
            entity.CapaRequestStatus = CapaRequestStatus.OPEN;

            var capaRequestLog = _mapper.Map<CapaRequestLog>(entity);
            capaRequestLog.Id = Guid.NewGuid();
            capaRequestLog.CreatedBy = userInfoToken.Id;
            capaRequestLog.CreatedDate = DateTime.UtcNow;
            capaRequestLog.LogStatus = LogStatus.Created;

            _capaRequestRepository.Add(entity);
            _capaRequestLogRepository.Add(capaRequestLog);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<CapaRequestDto>.Return500();
            }
            var entityDto = _mapper.Map<CapaRequestDto>(entity);
            return ServiceResponse<CapaRequestDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving CapaRequest");
            return ServiceResponse<CapaRequestDto>.Return500("Error while saving CapaRequest");
        }

    }
}
