﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddCapaReviewCommandHandler(
        ICapaReviewRepository _capaReviewRepository,
        ICapaRequestRepository _capaRequestRepository,
        IUserRepository _userRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddCapaReviewCommandHandler> _logger,
        UserInfoToken userInfoToken,
        ICapaActionRepository _capaActionRepository
        ) : IRequestHandler<AddCapaReviewCommand, ServiceResponse<CapaReviewDto>>
    {
        public async Task<ServiceResponse<CapaReviewDto>> Handle(AddCapaReviewCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExist = await _capaRequestRepository.FindAsync(request.CapaRequestId);
                if (entityExist == null)
                {
                    return ServiceResponse<CapaReviewDto>.Return404("CapaRequest not found");
                }
                if (request.Status == ReviewStatus.APPROVED || request.Status == ReviewStatus.REJECTED)
                {
                    var actionsStatus = await _capaActionRepository.All
                        .Where(c => c.CapaRequestId == request.CapaRequestId
                        && (c.Status == CapaActionStatus.All ||
                            c.Status == CapaActionStatus.PENDING ||
                            c.Status == CapaActionStatus.INPROGRESS)).CountAsync();
                    if (actionsStatus >= 1)
                    {
                        return ServiceResponse<CapaReviewDto>.Return409("All CAPA actions must be completed or rejected first.");
                    }
                    entityExist.CapaRequestStatus = request.Status == ReviewStatus.APPROVED ? CapaRequestStatus.CLOSED : CapaRequestStatus.REJECTED;
                    _capaRequestRepository.Update(entityExist);
                }

                var entity = _mapper.Map<CapaReview>(request);
                entity.ReviewedById = userInfoToken.Id;
                entity.ReviewedDate = DateTime.UtcNow;
                _capaReviewRepository.Add(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<CapaReviewDto>.Return500();
                }
                var user = await _userRepository.All.Where(c => c.Id == entity.ReviewedById).FirstOrDefaultAsync();
                if (user != null)
                {
                    entity.ReviewedBy = user;
                }
                var entityDto = _mapper.Map<CapaReviewDto>(entity);
                return ServiceResponse<CapaReviewDto>.ReturnResultWith201(entityDto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving CapaReview");
                return ServiceResponse<CapaReviewDto>.Return500("Error while saving CapaReview");
            }
        }
    }
}
