﻿using Amazon.Runtime.Internal.Util;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateComplaintActionCommandHandler(
     IComplaintActionRepository _complaintActionRepository,
     IMapper mapper,
     ILogger<UpdateComplaintActionCommandHandler> _logger,
     IUnitOfWork<QMSDbContext> _uow,
     IComplaintInvestigationRepository _complaintInvestigationRepository,
     IUserRepository _userRepository,
     IComplaintRepository _complaintRepository) : IRequestHandler<UpdateComplaintActionCommand, ServiceResponse<ComplaintActionDto>>
    {
        public async Task<ServiceResponse<ComplaintActionDto>> Handle(UpdateComplaintActionCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExit = await _complaintActionRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (entityExit == null)
                {
                    return ServiceResponse<ComplaintActionDto>.Return404("complaint action not found");
                }
                var complaint = await _complaintRepository.All.Where(c => c.Id == request.ComplaintId).FirstOrDefaultAsync();
                if (complaint == null)
                {
                    return ServiceResponse<ComplaintActionDto>.Return404("complaint not found");
                }
                if (request.Status == ComplaintStatusEnum.RESOLVED || request.Status == ComplaintStatusEnum.CLOSED)
                {
                    var investigationStatus = await _complaintInvestigationRepository.All
                        .Where(c => c.ComplaintId == entityExit.ComplaintId && (
                                    c.Status == ComplaintStatusEnum.OPEN ||
                                    c.Status == ComplaintStatusEnum.IN_PROGRESS ||
                                    c.Status == ComplaintStatusEnum.ESCALATED ||
                                    c.Status == ComplaintStatusEnum.PENDING)).CountAsync();
                    if (investigationStatus >= 1)
                    {
                        return ServiceResponse<ComplaintActionDto>.Return409("All Complaint investigation must be closed or rejected first.");
                    }
                }
                mapper.Map(request, entityExit);
                if (request.Status == ComplaintStatusEnum.CLOSED)
                {
                    entityExit.CompletedDate = DateTime.UtcNow;
                }
                _complaintActionRepository.Update(entityExit);
                if (await _uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<ComplaintActionDto>.Return500();
                }
                var user = await _userRepository.All.Where(c => c.Id == entityExit.ResponsibleId).FirstOrDefaultAsync();
                if (user != null)
                {
                    entityExit.Responsible = user;
                }
                var dto = mapper.Map<ComplaintActionDto>(entityExit);
                return ServiceResponse<ComplaintActionDto>.ReturnResultWith200(dto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "error while updating comaplaint Action");
                return ServiceResponse<ComplaintActionDto>.Return500("error while updating comaplaint Action");
            }
        }
    }
}
