﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadComplaintAttachmentChunkCommandHandler(
        IDocumentTokenRepository _documentTokenRepository,
        IComplaintAttchmentRepository _complaintAttchmentRepository,
        IComplaintAttachmentChunkRepository _complaintAttachmentChunkRepository,
        ILogger<DownloadRiskAttachmentChunkCommandHandler> _logger,
        IMediator _mediator) : IRequestHandler<DownloadComplaintAttachmentChunkCommand, ServiceResponse<DocumentDownload>>
    {
        public async Task<ServiceResponse<DocumentDownload>> Handle(DownloadComplaintAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var isTokenFound = await _documentTokenRepository.All.AnyAsync(c => c.DocumentId == request.Id && c.Token == request.Token);
                if (!isTokenFound)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Token not found");
                }
                var complaintAttachment = await _complaintAttchmentRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (complaintAttachment == null)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Complaint  attachment not found");
                }
                var complaintAttachmentChunks = await _complaintAttachmentChunkRepository.All.Where(c => c.ComplaintAttachmentId == complaintAttachment.Id).OrderBy(c => c.ChunkIndex).ToListAsync();
                if (complaintAttachmentChunks.Count <= 0)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Complaint attachment chunk not found");
                }
                var chunks = complaintAttachmentChunks.Select(c => new FileChunk
                {
                    ChunkIndex = c.ChunkIndex,
                    Url = c.Url,
                }).ToList();
                var fileDownloadRequestCommand = new FileDownloadRequestCommand
                {
                    StorageSettingId = complaintAttachment.StorageSettingId,
                    Key = complaintAttachment.Key,
                    IV = complaintAttachment.IV,
                    Extension = complaintAttachment.Extension,
                    Chunks = chunks,
                    FileName = complaintAttachment.FileName,
                };
                var result = await _mediator.Send(fileDownloadRequestCommand);
                return result;
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while getting Complaint  attachment");
                return ServiceResponse<DocumentDownload>.Return500("An unexpected fault happened. Try again later. ");
            }
        }
    }
}
