﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR.RiskManagement;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class MarkComplaintAttachmentChunksUploadedCommandHandler (
         IComplaintAttchmentRepository _complaintAttchmentRepository,
        IComplaintAttachmentChunkRepository  _complaintAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<MarkComplaintAttachmentChunksUploadedCommandHandler> _logger
        ) : IRequestHandler<MarkComplaintAttachmentChunksUploadedCommand, ServiceResponse<ComplaintAttachmentChunkStatus>>
    {
        public async Task<ServiceResponse<ComplaintAttachmentChunkStatus>> Handle(MarkComplaintAttachmentChunksUploadedCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var complaintAttachment = await _complaintAttchmentRepository.All
                    .Where(c => c.Id == request.ComplaintAttachmentId).FirstOrDefaultAsync();

                if (complaintAttachment == null)
                {
                    return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(404, "ComplaintAttachment not found.");
                }

                var complaintChunks = _complaintAttachmentChunkRepository
                    .All
                    .Where(c => c.ComplaintAttachmentId == complaintAttachment.Id)
                    .OrderByDescending(c => c.ChunkIndex)
                    .ToList();

                if (complaintChunks.Count == 0)
                {
                    return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(404, " ComplaintAttachment Chunks not found.");
                }

                if (complaintAttachment.TotalChunk == complaintChunks.Count())
                {
                    complaintAttachment.IsAllChunkUploaded = request.Status;

                     _complaintAttchmentRepository.Update(complaintAttachment);

                    if (await _uow.SaveAsync(cancellationToken) < 0)
                    {
                        return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding ComplaintAttachment");
                    }
                    var response = new ComplaintAttachmentChunkStatus { ComplaintAttachmentId = complaintAttachment.Id, Status = request.Status };
                    return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnResultWith200(response);
                }
                else
                {
                    if (complaintChunks.Count() > 0)
                    {
                        foreach (var documentChunk in complaintChunks)
                        {
                            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(complaintAttachment.StorageSettingId);
                            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                            await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                        }
                        _complaintAttachmentChunkRepository.RemoveRange(complaintChunks);
                    }

                    _complaintAttchmentRepository.Remove(complaintAttachment);
                    if (await _uow.SaveAsync() <= 0)
                    {
                        return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding ComplaintAttachment");
                    }

                    return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "Error while marking ComplaintAttachment chunks uploaded.");
                return ServiceResponse<ComplaintAttachmentChunkStatus>.ReturnFailed(500, "Error while marking ComplaintAttachment chunks uploaded.");
            }

        }
    }
}
