﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UploadComplaintAttachmentChunkCommandHandler (
         IComplaintAttchmentRepository _complaintAttchmentRepository,
        IComplaintAttachmentChunkRepository _complaintAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UploadComplaintAttachmentChunkCommandHandler> _logger) : IRequestHandler<UploadComplaintAttachmentChunkCommand, ServiceResponse<ComplaintAttachmentChunkDto>>
    {
        public async Task<ServiceResponse<ComplaintAttachmentChunkDto>> Handle(UploadComplaintAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                if (request.File == null)
                {
                    return ServiceResponse<ComplaintAttachmentChunkDto>.ReturnFailed(409, "Please select the file.");
                }

                var existComplaintAttachment = _complaintAttchmentRepository.Find(request.ComplaintAttachmentId);
                if (existComplaintAttachment == null)
                {
                    return ServiceResponse<ComplaintAttachmentChunkDto>.Return404("ComplaintAttachment not found");
                }
                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(existComplaintAttachment.StorageSettingId);

                var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);

                var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, existComplaintAttachment.Extension, existComplaintAttachment.Key, existComplaintAttachment.IV);

                if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
                {
                    return ServiceResponse<ComplaintAttachmentChunkDto>.Return422("Settings are not properly setup.");
                }
                var entity = new ComplaintAttachmentChunk()
                {
                    Id = Guid.NewGuid(),
                    ChunkIndex = request.ChunkIndex,
                    ComplaintAttachmentId = request.ComplaintAttachmentId,
                    Extension = request.Extension,
                    Size = request.Size,
                    Url = fileNameKeyValut.FileName
                };
                _complaintAttachmentChunkRepository.Add(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<ComplaintAttachmentChunkDto>.ReturnFailed(500, "Error While Added Document");
                }

                var entityDto = _mapper.Map<ComplaintAttachmentChunkDto>(entity);
                return ServiceResponse<ComplaintAttachmentChunkDto>.ReturnResultWith200(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while storing ComplaintAttachmentChunk");
                return ServiceResponse<ComplaintAttachmentChunkDto>.Return500("Error while storing ComplaintAttachmentChunk.");
            }
        }
    }
}
