﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using MediatR;
using Microsoft.Extensions.Logging;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class AddSessionMediaCommandHandler(
    ICourseSessionMediaRepository courseSessionMediaRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper _mapper,
    StorageServiceFactory _storeageServiceFactory,
    IStorageSettingRepository _storageSettingRepository,
    ILogger<AddSessionMediaCommandHandler> _logger) : IRequestHandler<AddSessionMediaCommand, ServiceResponse<CourseSessionMediaDto>>
{
    public async Task<ServiceResponse<CourseSessionMediaDto>> Handle(AddSessionMediaCommand request, CancellationToken cancellationToken)
    {
        try
        {
            if (request.MediaType == CourseSessionMediaType.FILE && request.Files == null)
            {
                return ServiceResponse<CourseSessionMediaDto>.ReturnFailed(409, "Please select the file.");
            }

            var entity = _mapper.Map<CourseSessionMedia>(request);

            if (request.MediaType == CourseSessionMediaType.FILE)
            {
                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(request.StorageSettingId);

                var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

                var fileNameKeyValut = await storageService.UploadFileAsync(request.Files, storeageSetting, request.Extension);

                if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
                {
                    return ServiceResponse<CourseSessionMediaDto>.Return422("Settings are not properly setup.");
                }
                entity.Url = fileNameKeyValut.FileName;
                entity.Key = fileNameKeyValut.Key;
                entity.IV = fileNameKeyValut.IV;
                entity.StorageType = storeageSetting.StorageType;
                entity.StorageSettingId = storeageSetting.Id;
            }

            entity.IsChunk = false;
            entity.IsAllChunkUploaded = true;
            courseSessionMediaRepository.Add(entity);

            if (await _uow.SaveAsync() <= 0)
            {
                return ServiceResponse<CourseSessionMediaDto>.ReturnFailed(500, "Error while storing media");
            }

            var entityDto = _mapper.Map<CourseSessionMediaDto>(entity);
            return ServiceResponse<CourseSessionMediaDto>.ReturnResultWith200(entityDto);
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error while storing media.");
            return ServiceResponse<CourseSessionMediaDto>.Return500("Error while storing media");
        }
    }

}
