﻿using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.MediatR;
using QualityManagement.Repository;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class MarkMediaChunksUploadedCommandHandler(
    ICourseSessionMediaRepository sessionMediaRepository,
    ICourseSessionMediaChunkRepository sessionMediaChunkRepository,
    IUnitOfWork<QMSDbContext> uow,
    IStorageSettingRepository _storageSettingRepository,
    StorageServiceFactory _storageServiceFactory,
    ILogger<MarkMediaChunksUploadedCommandHandler> _logger)
    : IRequestHandler<MarkMediaChunksUploadedCommand, ServiceResponse<SessionMediaChunkStatus>>
{
    public async Task<ServiceResponse<SessionMediaChunkStatus>> Handle(MarkMediaChunksUploadedCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var sessionMedia = await sessionMediaRepository.All.Where(c => c.Id == request.CourseSessionMediaId).FirstOrDefaultAsync();

            if (sessionMedia == null)
            {
                return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(404, "Media not found.");
            }

            var mediaChunks = sessionMediaChunkRepository
                .All
                .Where(c => c.CourseSessionMediaId == sessionMedia.Id)
                .OrderByDescending(c => c.ChunkIndex)
                .ToList();

            if (mediaChunks.Count == 0)
            {
                return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(404, "DocMediaument Chunks not found.");
            }


            if (mediaChunks[0].TotalChunk == mediaChunks.Count())
            {
                sessionMedia.IsAllChunkUploaded = request.status;

                sessionMediaRepository.Update(sessionMedia);

                if (await uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(500, "Error While Adding Media");
                }
                var response = new SessionMediaChunkStatus { CourseSessionMediaId = sessionMedia.Id, Status = request.status };
                return ServiceResponse<SessionMediaChunkStatus>.ReturnResultWith200(response);
            }
            else
            {
                if (mediaChunks.Count() > 0)
                {
                    foreach (var documentChunk in mediaChunks)
                    {
                        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(sessionMedia.StorageSettingId);
                        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                        await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                    }
                    sessionMediaChunkRepository.RemoveRange(mediaChunks);
                }


                sessionMediaRepository.Remove(sessionMedia);
                if (await uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(500, "Error While Adding Media");
                }

                return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
            }
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error while marking media chunks uploaded.");
            return ServiceResponse<SessionMediaChunkStatus>.ReturnFailed(500, "Error while marking media chunks uploaded.");
        }
    }
}