﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class UpdateQuizeQuestionCommandHandler(
    IQuizQuestionRepository quizQuestionRepository,
 IQuizQuestionOptionRepository quizQuestionOptionRepository,
    ILogger<UpdateQuizeQuestionCommandHandler> logger,
    IMapper mapper,
    IUnitOfWork<QMSDbContext> unitOfWork,
    UserInfoToken userInfoToken)
    : IRequestHandler<UpdateQuizeQuestionCommand, ServiceResponse<QuizQuestionDto>>
{
    public async Task<ServiceResponse<QuizQuestionDto>> Handle(UpdateQuizeQuestionCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var question = await quizQuestionRepository.All.FirstOrDefaultAsync(c => c.Id == request.Id);
            var options = await quizQuestionOptionRepository.All.Where(c => c.QuizQuestionId == request.Id).ToListAsync();

            if (question == null)
            {
                logger.LogWarning("Quiz question not found");
                return ServiceResponse<QuizQuestionDto>.Return404("Quiz question not found");
            }

            question.QuestionText = request.QuestionText;
            question.QuestionType = request.QuestionType;
            question.OrderNo = request.OrderNo;

            var optionsToAdd = request.QuizQuestionOptions.Where(c => !c.Id.HasValue).ToList();
            var optionsToUpdate = request.QuizQuestionOptions.Where(c => c.Id.HasValue).ToList();

            var optionIdsFromUI = request.QuizQuestionOptions.Where(c => c.Id.HasValue).Select(cs => cs.Id);
            var optionsToDelete = options.Where(c => !optionIdsFromUI.Contains(c.Id)).ToList();

            foreach (var option in optionsToUpdate)
            {
                var optionToUpdate = options.FirstOrDefault(c => c.Id == option.Id);
                if (optionToUpdate != null)
                {
                    optionToUpdate.OptionText = option.OptionText;
                    optionToUpdate.IsCorrect = option.IsCorrect;
                    quizQuestionOptionRepository.Update(optionToUpdate);
                }
            }

            foreach (var option in optionsToAdd)
            {
                quizQuestionOptionRepository.Add(new QuizQuestionOption
                {
                    OptionText = option.OptionText,
                    IsCorrect = option.IsCorrect,
                    QuizQuestionId = question.Id,
                });
            }

            foreach (var option in optionsToDelete)
            {
                option.DeletedBy = userInfoToken.Id;
                option.IsDeleted = true;
                quizQuestionOptionRepository.Update(option);
            }

            quizQuestionRepository.Update(question);
            if (await unitOfWork.SaveAsync() <= 0)
            {
                logger.LogWarning("Error while updating quiz question");
                return ServiceResponse<QuizQuestionDto>.Return500("Error while updating quiz question");
            }

            question = await quizQuestionRepository.All
                .Include(c => c.QuizQuestionOptions)
                .FirstOrDefaultAsync(c => c.Id == request.Id);

            var questionDto = mapper.Map<QuizQuestionDto>(question);
            return ServiceResponse<QuizQuestionDto>.ReturnResultWith200(questionDto);
        }
        catch (System.Exception ex)
        {
            logger.LogError(ex, "Error while updating quiz question");
            return ServiceResponse<QuizQuestionDto>.Return500("Error while updating quiz question");
        }
    }
}
