﻿using AutoMapper;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using MediatR;
using Microsoft.Extensions.Logging;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;

namespace QualityManagement.MediatR;
public class AddCourseSessionCommandHandler(
    ICourseSessionRepository courseSessionRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<AddCourseSessionCommandHandler> logger,
    IMapper mapper)
    : IRequestHandler<AddCourseSessionCommand, ServiceResponse<CourseSessionDto>>
{
    public async Task<ServiceResponse<CourseSessionDto>> Handle(AddCourseSessionCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existingCourseSession = await courseSessionRepository.All
                .FirstOrDefaultAsync(d => d.CourseId == request.CourseId && d.Title.ToLower() == request.Title.ToLower());

            if (existingCourseSession != null)
            {
                return ServiceResponse<CourseSessionDto>.Return409("Course session with the same title already exists.");
            }

            var courseSession = mapper.Map<CourseSession>(request);
            courseSessionRepository.Add(courseSession);
            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<CourseSessionDto>.Return500("Error while saving course session.");
            }
            return ServiceResponse<CourseSessionDto>.ReturnResultWith200(mapper.Map<CourseSessionDto>(courseSession));
        }
        catch (System.Exception ex)
        {
            logger.LogError(ex, "Error while saving course session.");
            return ServiceResponse<CourseSessionDto>.Return500("Error while saving course session.");
        }
    }
}
