﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class AddCourseCommandHandler(
    ICourseRepository courseRepository,
    IMapper mapper,
    ILogger<AddCourseCommandHandler> logger,
    IUnitOfWork<QMSDbContext> unitOfWork)
    : IRequestHandler<AddCourseCommand, ServiceResponse<CourseDto>>
{
    public async Task<ServiceResponse<CourseDto>> Handle(AddCourseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existingCourse = await courseRepository.All
                .Where(c => c.Title.ToLower() == request.Title.ToLower())
                .FirstOrDefaultAsync();

            if (existingCourse != null)
            {
                return ServiceResponse<CourseDto>.Return409("Course with the same title already exists.");
            }

            var course = mapper.Map<Course>(request);
            course.CourseStatus = CourseStatus.DRAFT;
            courseRepository.Add(course);

            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<CourseDto>.Return500();
            }

            return ServiceResponse<CourseDto>.ReturnResultWith201(mapper.Map<CourseDto>(course));
        }
        catch (System.Exception ex)
        {
            logger.LogError(ex, "Error while saving course");
            return ServiceResponse<CourseDto>.Return500("Error while saving course");
        }
    }
}
