﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.AspNetCore.SignalR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class DeleteDocumentRolePermissionCommandHandler(
      IDocumentRolePermissionRepository _documentRolePermissionRepository,
      IUnitOfWork<QMSDbContext> _uow,
      UserInfoToken _userInfo,
      IDocumentAuditTrailRepository _documentAuditTrailRepository,
      ICategoryRepository _categoryRepository,
      IHubContext<UserHub, IHubClient> _hubContext,
      IConnectionMappingRepository _connectionMappingRepository,
      ILogger<AddDocumentRolePermissionCommandHandler> _logger,
      IDocumentRepository _documentRepository) : IRequestHandler<DeleteDocumentRolePermissionCommand, DocumentRolePermissionDto>
{

    public async Task<DocumentRolePermissionDto> Handle(DeleteDocumentRolePermissionCommand request, CancellationToken cancellationToken)
    {
        var entity = await _documentRolePermissionRepository.FindAsync(request.Id);
        if (entity == null)
        {
            var errorDto = new DocumentRolePermissionDto
            {
                StatusCode = 404,
                Messages = new List<string> { "Not Found" }
            };
            return errorDto;
        }
        var documentAudit = new DocumentAuditTrail()
        {
            DocumentId = entity.DocumentId,
            CreatedBy = _userInfo.Id,
            CreatedDate = DateTime.UtcNow,
            OperationName = DocumentOperation.Removed_Permission,
            AssignToRoleId = entity.RoleId
        };
        _documentAuditTrailRepository.Add(documentAudit);

        _documentRolePermissionRepository.Remove(entity);
        var documentInfo = _documentRepository.Find(entity.DocumentId);
        await _documentRepository.UpdateDocumentSharingFlagAsync(new List<Guid> { documentInfo.CategoryId });
        if (await _uow.SaveAsync() <= -1)
        {
            var errorDto = new DocumentRolePermissionDto
            {
                StatusCode = 500,
                Messages = new List<string> { "An unexpected fault happened. Try again later." }
            };
            return errorDto;
        }

        var category = _categoryRepository.All.Where(c => c.Id == documentInfo.CategoryId).FirstOrDefault();
        try
        {
            var onlineUsers = _connectionMappingRepository.GetAllUsersExceptThis(new SignlarUser { Id = _userInfo.Id.ToString() });
            if (onlineUsers.Count() > 0)
            {
                var userInfo = _connectionMappingRepository.GetUserInfoById(_userInfo.Id);
                if (userInfo != null)
                {
                    await _hubContext.Clients.AllExcept(new List<string> { userInfo.ConnectionId }).SendNotificationFolderChange(category.ParentId);
                }
                else
                {
                    await _hubContext.Clients.All.SendNotificationFolderChange(category.ParentId);
                }
            }
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "SignalR Error");
        }

        return new DocumentRolePermissionDto
        {
            StatusCode = 200,
            Messages = new List<string> { "Permission Deleted Successfully." }
        };
    }
}
