﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class CreateEmployeeCourseSessionQuizAnswerCommandHandler( 
    IQuizQuestionOptionRepository quizQuestionOptionRepository,
    IEmployeeCourseSessionQuizRepository employeeCourseSessionQuizRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<CreateEmployeeCourseSessionQuizAnswerCommandHandler> logger,
    IMapper mapper,
    IEmployeeCourseRepository employeeCourseRepository,
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    ICourseSessionRepository courseSessionRepository)
    : IRequestHandler<CreateEmployeeCourseSessionQuizAnswerCommand, ServiceResponse<EmployeeCourseSessionQuizDto>>
{
    public async Task<ServiceResponse<EmployeeCourseSessionQuizDto>> Handle(CreateEmployeeCourseSessionQuizAnswerCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourse = employeeCourseRepository.All
                .Include(d => d.EmployeeCourseSessions)
                    .ThenInclude(s => s.CourseSession)
                        .ThenInclude(c => c.CourseSessionQuiz)
                .FirstOrDefault(x => x.Link == request.Link);

            if (employeeCourse == null)
            {
                return ServiceResponse<EmployeeCourseSessionQuizDto>.Return409("Employee course is not assigned");
            }

            var courseSession = await courseSessionRepository
                    .AllIncluding(d => d.CourseSessionQuiz)
                    .FirstOrDefaultAsync(d => d.Id == request.CourseSessionId);
            if (courseSession == null)
            {
                return ServiceResponse<EmployeeCourseSessionQuizDto>.Return404("Course session not found.");
            }

            var isEmployeeCourseSessionExists = true;
            var employeeCourseSession = employeeCourse.EmployeeCourseSessions
            .FirstOrDefault(x => x.CourseSessionId == request.CourseSessionId);

            if (employeeCourseSession == null)
            {
                isEmployeeCourseSessionExists = false;
                employeeCourseSession = new EmployeeCourseSession
                {
                    Id = Guid.NewGuid(),
                    CourseSessionId = request.CourseSessionId,
                    EmployeeCourseId = employeeCourse.Id,
                    EmployeeId = employeeCourse.EmployeeId
                };
            }
            else
            {
                //delexisting quiz if it exists
                var existingQuiz = await employeeCourseSessionQuizRepository.All.FirstOrDefaultAsync(c=>c.EmployeeCourseSessionId == employeeCourseSession.Id);

                if (existingQuiz != null)
                {
                    employeeCourseSessionQuizRepository.Remove(existingQuiz);
                }
            }

            var employeeCousseSessionQuiz = new Data.EmployeeCourseSessionQuiz
            {
                Id = Guid.NewGuid(),
                IsPassed = false,
                ScorePercentage = 0,
                StartDate = DateTime.UtcNow,
                SubmittedDate = DateTime.UtcNow,
                EmployeeCourseSessionId = employeeCourseSession.Id,
            };

            var questionIds = request.EmployeeCourseSessionQuizQuestions.Select(d => d.QuestionId).ToList();
            var quizQuestionOptions = await quizQuestionOptionRepository
                .All
                .Where(x => questionIds.Contains(x.QuizQuestionId))
                .ToListAsync();

            var passingScore = courseSession.CourseSessionQuiz.PassingScore;

            var answerToAdd = new List<EmployeeCourseSessionQuizAnswer>();
            foreach (var answer in request.EmployeeCourseSessionQuizQuestions)
            {
                var correctOptions = quizQuestionOptions
                    .Where(q => q.QuizQuestionId == answer.QuestionId && q.IsCorrect)
                    .Select(q => q.Id)
                    .ToList();

                var quizAnswer = new EmployeeCourseSessionQuizAnswer
                {
                    EmployeeCourseSessionQuizId = employeeCousseSessionQuiz.Id,
                    QuestionId = answer.QuestionId,
                    IsCorrect = answer.SelectedOptionIds.Intersect(correctOptions).Any(),
                    EmployeeCourseSessionQuizAnswerOptions = answer.SelectedOptionIds.Select(d => new EmployeeCourseSessionQuizAnswerOption
                    {
                        OptionId = d,
                    }).ToList()
                };
                answerToAdd.Add(quizAnswer);
            }

            employeeCousseSessionQuiz.IsPassed = answerToAdd.Count(x => x.IsCorrect) >= passingScore;
            employeeCousseSessionQuiz.SubmittedDate = DateTime.UtcNow;
            employeeCousseSessionQuiz.ScorePercentage = (decimal)answerToAdd.Count(x => x.IsCorrect) / questionIds.Count * 100;
            employeeCousseSessionQuiz.EmployeeCourseSessionQuizAnswers = answerToAdd;

            employeeCourseSession.CompletedDate = DateTime.UtcNow;
            employeeCourseSession.IsPassed = employeeCousseSessionQuiz.IsPassed;

            if (isEmployeeCourseSessionExists)
            {
                employeeCourseSessionRepository.Update(employeeCourseSession);
            }
            else
            {
                employeeCourseSessionRepository.Add(employeeCourseSession);
            }

            employeeCourseSessionQuizRepository.Add(employeeCousseSessionQuiz);
            var courseSessions = await courseSessionRepository.All.Where(c=>c.CourseId==employeeCourse.CourseId).CountAsync();
            var EmployeeSessionCount = employeeCourse.EmployeeCourseSessions.Count;

            bool allSessionsCompleted = employeeCourse.EmployeeCourseSessions.All(s => s.IsPassed);

            if (allSessionsCompleted)
            {
                employeeCourse.Status = EmployeeCourseStatus.Completed;
                employeeCourse.CompletedDate = DateTime.UtcNow;
            }
            else
            {
                employeeCourse.Status = EmployeeCourseStatus.InProgress;
            }

            employeeCourseRepository.Update(employeeCourse);

            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<EmployeeCourseSessionQuizDto>.Return500("Failed to create quiz answer.");
            }

            var responseDto = mapper.Map<EmployeeCourseSessionQuizDto>(employeeCousseSessionQuiz);
            return ServiceResponse<EmployeeCourseSessionQuizDto>.ReturnResultWith201(responseDto);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while creating employee course session quiz answer.");
            return ServiceResponse<EmployeeCourseSessionQuizDto>.Return500("Failed to create quiz answer.");
        }
    }
}
