﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class GetEmployeeCourseSessionByIdCommandHandler(
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    ILogger<GetEmployeeCourseSessionByIdCommandHandler> logger)
    : IRequestHandler<GetEmployeeCourseSessionByIdCommand, ServiceResponse<EmployeeCourseSessionDetailDto>>
{
    public async Task<ServiceResponse<EmployeeCourseSessionDetailDto>> Handle(GetEmployeeCourseSessionByIdCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourseSession = await employeeCourseSessionRepository
                .All
                .Include(d => d.Employee)
                .Include(d => d.CourseSession)
                    .ThenInclude(d => d.Course)
                .Include(d => d.CourseSession)
                    .ThenInclude(d => d.CourseSessionMedias)
                .Where(x => x.Id == request.Id)
                .Select(employeeCourseSession => new EmployeeCourseSessionDetailDto
                {
                    Id = employeeCourseSession.Id,
                    CourseName = employeeCourseSession.CourseSession.Course.Title,
                    SessionName = employeeCourseSession.CourseSession.Title,
                    SessionDescription = employeeCourseSession.CourseSession.Description,
                    EmployeeId = employeeCourseSession.EmployeeId,
                    CourseSessionId = employeeCourseSession.CourseSessionId,
                    CompletedDate = employeeCourseSession.CompletedDate,
                    EmployeeName = $"{employeeCourseSession.Employee.FirstName} {employeeCourseSession.Employee.LastName}",
                    CourseSessionMedias = employeeCourseSession.CourseSession.CourseSessionMedias.Select(m => new CourseSessionMediaDto
                    {
                        Id = m.Id,
                        MediaType = m.MediaType,
                        Name = m.Name,
                        Url = m.Url,
                        Description = m.Description,
                        Extension = m.Extension,
                        IsChunk = m.IsChunk,
                    }).ToList(),

                }).FirstOrDefaultAsync();

            if (employeeCourseSession == null)
            {
                return ServiceResponse<EmployeeCourseSessionDetailDto>.Return404("Employee Course Session not found.");
            }

            return ServiceResponse<EmployeeCourseSessionDetailDto>.ReturnResultWith200(employeeCourseSession);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while getting employee course session.");
            return ServiceResponse<EmployeeCourseSessionDetailDto>.Return500("An error occurred while getting employee course session.");
        }
    }
}