﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class GetEmployeeCourseSessionByLinkCommandHandler(
    IEmployeeCourseSessionRepository employeeCourseSessionRepository,
    ILogger<GetEmployeeCourseSessionByLinkCommandHandler> logger)
    : IRequestHandler<GetEmployeeCourseSessionByLinkCommand, ServiceResponse<EmployeeCourseSessionDetailDto>>
{
    public async Task<ServiceResponse<EmployeeCourseSessionDetailDto>> Handle(GetEmployeeCourseSessionByLinkCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var employeeCourseSession = await employeeCourseSessionRepository
                .All
                .Include(d => d.Employee)
                .Include(d => d.CourseSession)
                    .ThenInclude(d => d.Course)
                .Include(d => d.CourseSession)
                    .ThenInclude(d => d.CourseSessionMedias)
                .Include(d => d.CourseSession)
                    .ThenInclude(d => d.CourseSessionQuiz)
                .Select(employeeCourseSession => new EmployeeCourseSessionDetailDto
                {
                    Id = employeeCourseSession.Id,
                    CourseName = employeeCourseSession.CourseSession.Course.Title,
                    SessionName = employeeCourseSession.CourseSession.Title,
                    SessionDescription = employeeCourseSession.CourseSession.Description,
                    EmployeeId = employeeCourseSession.EmployeeId,
                    UserId = employeeCourseSession.Employee.Id,
                    CourseSessionId = employeeCourseSession.CourseSessionId,
                    EmployeeName = $"{employeeCourseSession.Employee.FirstName} {employeeCourseSession.Employee.LastName}",
                    CompletedDate = employeeCourseSession.CompletedDate,
                    IsPassed = employeeCourseSession.IsPassed,
                    HasQuiz = employeeCourseSession.CourseSession.CourseSessionQuiz != null,
                    CourseSessionMedias = employeeCourseSession.CourseSession.CourseSessionMedias.Select(m => new CourseSessionMediaDto
                    {
                        Id = m.Id,
                        MediaType = m.MediaType,
                        Name = m.Name,
                        Url = m.Url,
                        Description = m.Description,
                        Extension = m.Extension,
                        IsChunk = m.IsChunk,
                    }).ToList(),

                }).FirstOrDefaultAsync();

            if (employeeCourseSession == null)
            {
                return ServiceResponse<EmployeeCourseSessionDetailDto>.Return404("Employee Course Session not found.");
            }

            if (!employeeCourseSession.UserId.HasValue)
            {
                return ServiceResponse<EmployeeCourseSessionDetailDto>.Return404("Employee registration not completed.");
            }

            return ServiceResponse<EmployeeCourseSessionDetailDto>.ReturnResultWith200(employeeCourseSession);
        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error occurred while getting employee course session.");
            return ServiceResponse<EmployeeCourseSessionDetailDto>.Return500("An error occurred while getting employee course session.");
        }
    }
}
