﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddNonConformanceRespAttachmentCommandHandler(
        INonConformanceResponseRepository _nonConformanceResponseRepository,
        INonConformanceRespAttachmentRepository _nonConformanceRespAttachmentRepository,
        IStorageSettingRepository _storageSettingRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddNonConformanceResponseCommandHandler> _logger) : IRequestHandler<AddNonConformanceRespAttachmentCommand, ServiceResponse<NonConformanceRespAttachmentDto>>
    {
        public async Task<ServiceResponse<NonConformanceRespAttachmentDto>> Handle(AddNonConformanceRespAttachmentCommand request, CancellationToken cancellationToken)
        {
            try
            {

                var existNcResponse = await _nonConformanceResponseRepository.FindAsync(request.NonConformanceResponseId);
                if (existNcResponse == null)
                {
                    return ServiceResponse<NonConformanceRespAttachmentDto>.Return404("NonConformanceResponse not found");
                }

                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(null);
                var keyValut = KeyGenerator.GenerateKeyAndIV();
                var entity = _mapper.Map<NonConformanceRespAttachment>(request);
                entity.Key = storeageSetting.EnableEncryption ? keyValut.Item1 : null;
                entity.IV = storeageSetting.EnableEncryption ? keyValut.Item2 : null;
                entity.StorageType = storeageSetting.StorageType;
                entity.StorageSettingId = storeageSetting.Id;

                _nonConformanceRespAttachmentRepository.Add(entity);

                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NonConformanceRespAttachmentDto>.Return500();
                }
                var entityDto = _mapper.Map<NonConformanceRespAttachmentDto>(entity);
                return ServiceResponse<NonConformanceRespAttachmentDto>.ReturnResultWith201(entityDto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving NonConformanceResponseAttachment");
                return ServiceResponse<NonConformanceRespAttachmentDto>.Return500("Error while saving NonConformanceResponseAttachment");
            }
        }
    }

}


