﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateNonConformanceResponseCommandHandler(
        INonConformanceResponseRepository _nonConformanceResponseRepository,
        IMapper _mapper,
        ILogger<UpdateNonConformanceResponseCommandHandler> _logger,
        IUnitOfWork<QMSDbContext> _uow,
         UserInfoToken _userInfoToken,
    IUserRepository _userRepository,
    ISendEmailRepository _sendEmailRepository,
    IUserNotificationRepository _userNotificationRepository) : IRequestHandler<UpdateNonConformanceResponseCommand, ServiceResponse<NonConformanceResponseDto>>
    {
        public async Task<ServiceResponse<NonConformanceResponseDto>> Handle(UpdateNonConformanceResponseCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var existEntity = await _nonConformanceResponseRepository.FindAsync(request.Id);
                if (existEntity == null)
                {
                    return ServiceResponse<NonConformanceResponseDto>.Return404("NonConformance not found");
                }
                // Send Email If Responsible Person not same
                if (request.ResponsiblePersonId != existEntity.ResponsiblePersonId)
                {
                    var userIds = new[] { _userInfoToken.Id, request.ResponsiblePersonId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var user = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var responsibleUser = users.FirstOrDefault(u => u.Id == request.ResponsiblePersonId);
                    try
                    {
                        if (responsibleUser != null)
                        {
                            var mainTitle = "Non-Conformance Response Assignment Notification";
                            var subject = "Non-Conformance Response Assignment";
                            var emailMessage = $"You have been assigned as the responsible person for a Non-Conformance Response : <b>{request.Title}</b>.  Please log in to the system to review the details, provide your response, and complete the required actions before the due date.";
                            _sendEmailRepository.AddAsignmentEmails(new SendEmail
                            {
                                Email = responsibleUser.Email,
                                FromName = user.FirstName + ' ' + user.LastName,
                                ToName = responsibleUser.FirstName + ' ' + responsibleUser.LastName,
                                CreatedBy = user.Id,
                                CreatedDate = DateTime.UtcNow,
                            }, emailMessage, mainTitle, subject);

                            // Sent Notification
                            var notificationDto = new UserNotificationDto
                            {
                                Id = existEntity.Id,
                                UserId = request.ResponsiblePersonId.Value,
                                NotificationsType = NotificationsType.NC_RESPONSE,
                                Title = request.Title,
                            };
                            _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                        }
                    } 
                    catch (System.Exception ex)
                    {
                        _logger.LogError(ex, "email does not sent");
                    }
                }
                _mapper.Map(request, existEntity);
                existEntity.NonConformanceId = existEntity.NonConformanceId;

                _nonConformanceResponseRepository.Update(existEntity);

                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NonConformanceResponseDto>.Return500();
                }

                var entityDto = _mapper.Map<NonConformanceResponseDto>(existEntity);
                return ServiceResponse<NonConformanceResponseDto>.ReturnResultWith200(entityDto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while updating NonConformanceResponse");
                return ServiceResponse<NonConformanceResponseDto>.Return500("Error while updating NonConformanceResponse");
            }

        }
    }
}
