﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UploadNonConformanceRespAttachmenttChunkCommandHandler(
        INonConformanceResponseAttachmentChunkRepository _nonConformanceResponseAttachmentChunkRepository,
        INonConformanceRespAttachmentRepository nonConformanceRespAttachmentRepository,
  IStorageSettingRepository _storageSettingRepository,
   StorageServiceFactory _storeageServiceFactory,
   IUnitOfWork<QMSDbContext> _uow,
   IMapper _mapper,
   ILogger<UploadNonConformanceRespAttachmenttChunkCommandHandler> logger
   ) : IRequestHandler<UploadNonConformanceRespAttachmenttChunkCommand, ServiceResponse<NonConformanceRespAttachmentChunkDto>>
    {
        public async Task<ServiceResponse<NonConformanceRespAttachmentChunkDto>> Handle(UploadNonConformanceRespAttachmenttChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                if (request.File == null)
                {
                    return ServiceResponse<NonConformanceRespAttachmentChunkDto>.ReturnFailed(409, "Please select the file.");
                }

                var ncAttach = nonConformanceRespAttachmentRepository.Find(request.NonConformanceRespAttachmentId);
                if (ncAttach == null)
                {
                    return ServiceResponse<NonConformanceRespAttachmentChunkDto>.ReturnFailed(404, "Document not found.");
                }

                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(ncAttach.StorageSettingId);

                var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

                var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, ncAttach.Extension, ncAttach.Key, ncAttach.IV);

                if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
                {
                    return ServiceResponse<NonConformanceRespAttachmentChunkDto>.Return422("Settings are not properly setup.");
                }

                var entity = new NonConformanceRespAttachmentChunk()
                {
                    Id = Guid.NewGuid(),
                    ChunkIndex = request.ChunkIndex,
                    NonConformanceRespAttachmentId = request.NonConformanceRespAttachmentId,
                    Extension = request.Extension,
                    Size = request.Size,
                    Url = fileNameKeyValut.FileName
                };

                _nonConformanceResponseAttachmentChunkRepository.Add(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<NonConformanceRespAttachmentChunkDto>.ReturnFailed(500, "Error While Added Document");
                }

                var entityDto = _mapper.Map<NonConformanceRespAttachmentChunkDto>(entity);
                return ServiceResponse<NonConformanceRespAttachmentChunkDto>.ReturnResultWith200(entityDto);
            }
            catch (Exception ex)
            {
                logger.LogError(ex, "Error while storing NonConformanceRespAttachmentChunk.");
                return ServiceResponse<NonConformanceRespAttachmentChunkDto>.Return500("Error while storing NonConformanceRespAttachmentChunk");
            }

        }
    }

}
