﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Entities.RiskManagment;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class UpdateRiskCommandHandler(
       IRiskRepository _riskRepository,
        IRiskMatrixLevelRepository _riskMatrixLevelRepository,
        IRiskLogRepository _riskLogRepository,
        IRiskControlRepository _riskControlRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UpdateRiskCommandHandler> _logger,
        IUserRepository _userRepository,
        ISendEmailRepository _sendEmailRepository,
        UserInfoToken _userInfoToken,
        IUserNotificationRepository _userNotificationRepository) : IRequestHandler<UpdateRiskCommand, ServiceResponse<RiskDto>>
    {
        public async Task<ServiceResponse<RiskDto>> Handle(UpdateRiskCommand request, CancellationToken cancellationToken)
        {

            try
            {
                var entityExist = await _riskRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (entityExist == null)
                {
                    return ServiceResponse<RiskDto>.Return404("risk not found");
                }
                if(request.Status == RiskStatusEnum.Closed || request.Status == RiskStatusEnum.Reviewed)
                {
                    var isPendingControl = await _riskControlRepository.All.Where(c=>c.RiskId==entityExist.Id)
                        .AnyAsync(c => c.Status == RiskStatusEnum.Open ||
                                  c.Status == RiskStatusEnum.Monitoring ||
                                  c.Status == RiskStatusEnum.Escalated);
                    if (isPendingControl)
                    {
                        return ServiceResponse<RiskDto>.Return409("All Risk Control must be Closed or Reviewed first.");
                    }
                }
                // If AssignedTo change Then Send Email
                if (request.AssignedToId != entityExist.AssignedToId)
                {
                    try
                    {
                        var userIds = new[] { _userInfoToken.Id, request.AssignedToId };
                        var users = await _userRepository.All
                            .Where(u => userIds.Contains(u.Id))
                            .ToListAsync();

                        var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                        var asignedUser = users.FirstOrDefault(u => u.Id == request.AssignedToId);
                        if (asignedUser != null)
                        {
                            var mainTitle = "Risk Assignment Notification";
                            var subject = "Risk Assignment";
                            var emailMessage = $"You have been assigned as the responsible person for a Risk : <b>{request.Title}</b>. Please log in to the system to review the risk details, assess its impact, and take the necessary actions to mitigate or manage it before the due date.";
                            _sendEmailRepository.AddAsignmentEmails(new SendEmail
                            {
                                Email = asignedUser.Email,
                                FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                                ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                                CreatedBy = fromUser.Id,
                                CreatedDate = DateTime.UtcNow,
                            }, emailMessage, mainTitle, subject);

                            // Sent Notification
                            var notificationDto = new UserNotificationDto
                            {
                                Id = entityExist.Id,
                                UserId = request.AssignedToId,
                                NotificationsType = NotificationsType.RISK,
                                Title = request.Title
                            };
                            _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                        }
                    }
                    catch (System.Exception ex)
                    {
                        _logger.LogError(ex, "email does not sent");
                    }
                }
                entityExist = _mapper.Map(request, entityExist);
                int riskScore = request.Likelihood * request.Severity;
                var riskLevel = await _riskMatrixLevelRepository
                    .All
                    .Where(c => c.ScoreMin <= riskScore && c.ScoreMax >= riskScore).FirstOrDefaultAsync();
                entityExist.RiskScore = riskScore;
                entityExist.RiskLevel = riskLevel != null ? riskLevel.RiskLevel : " ";

                var riskLog = _mapper.Map<RiskLog>(entityExist);
                riskLog.Id = Guid.NewGuid();
                riskLog.LogStatus = LogStatus.Updated;

                _riskRepository.Update(entityExist);
                _riskLogRepository.Add(riskLog);
                
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskDto>.Return500();
                }

                entityExist = await _riskRepository
                    .AllIncluding(c => c.RiskCategory, c => c.AssignedTo, c => c.CreatedUser)
                    .Where(c => c.Id == request.Id)
                    .FirstOrDefaultAsync();

                var entityDto = _mapper.Map<RiskDto>(entityExist);
                return ServiceResponse<RiskDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving risk");
                return ServiceResponse<RiskDto>.Return500("Error while saving risk");
            }
        }
    }
}
