﻿using System;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;

namespace QualityManagement.MediatR.RiskManagement;

public class AddRiskAttachmentCommandHandler(
    IRiskRepository _riskRepository,
    IRiskAttachmentRepository _riskAttachmentRepository,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddRiskAttachmentCommandHandler> _logger,
    UserInfoToken userInfoToken,
    IStorageSettingRepository _storageSettingRepository) : IRequestHandler<AddRiskAttachmentCommand, ServiceResponse<RiskAttachmentDto>>
{
    public async Task<ServiceResponse<RiskAttachmentDto>> Handle(AddRiskAttachmentCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existCapaRequest = await _riskRepository.FindAsync(request.RiskId);
            if (existCapaRequest == null)
            {
                return ServiceResponse<RiskAttachmentDto>.Return404("risk not found");
            }
            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(null);
            var keyValut = KeyGenerator.GenerateKeyAndIV();
            var entity = _mapper.Map<RiskAttachment>(request);
            entity.Key = storeageSetting.EnableEncryption ? keyValut.Item1 : null;
            entity.IV = storeageSetting.EnableEncryption ? keyValut.Item2 : null;
            entity.StorageSettingId = storeageSetting.Id;
            entity.UploadedById = userInfoToken.Id;
            entity.UploadedDate = DateTime.UtcNow;

            _riskAttachmentRepository.Add(entity);

            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<RiskAttachmentDto>.Return500();
            }
            var entityDto = _mapper.Map<RiskAttachmentDto>(entity);
            return ServiceResponse<RiskAttachmentDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving RiskAttachment");
            return ServiceResponse<RiskAttachmentDto>.Return500("Error while saving RiskAttachment");
        }
    }
}
