﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadRiskAttachmentCommandHandler(
          IRiskAttachmentRepository _riskAttachmentRepository,
         IRiskAttachmentChunkRepository _riskAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadRiskAttachmentCommand, ServiceResponse<RiskAttachmentDownload>>
    {
        public async Task<ServiceResponse<RiskAttachmentDownload>> Handle(DownloadRiskAttachmentCommand request, CancellationToken cancellationToken)
        {
            var riskAttachment = _riskAttachmentRepository.Find(request.RiskAttachmentId);
            if (riskAttachment == null)
            {
                return ServiceResponse<RiskAttachmentDownload>.ReturnFailed(404, "RiskAttachment is not found");
            }

            var riskAttachmentChunk = await _riskAttachmentChunkRepository.All
                .Where(c => c.RiskAttachmentId == request.RiskAttachmentId && c.ChunkIndex == request.ChunkIndex)
                .FirstOrDefaultAsync();

            if (riskAttachmentChunk == null)
            {
                return ServiceResponse<RiskAttachmentDownload>.ReturnFailed(404, "RiskAttachment Chunk is not found");
            }

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(riskAttachment.StorageSettingId);

            if (storeageSetting == null)
            {
                return ServiceResponse<RiskAttachmentDownload>.ReturnFailed(404, "Storage setting not found");
            }
            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
            var fileResult = await storageService.DownloadFileAsync(riskAttachmentChunk.Url, storeageSetting.JsonValue, riskAttachment.Key, riskAttachment.IV);
            if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
            {
                var extension = riskAttachmentChunk.Extension;
                if (!extension.StartsWith("."))
                    extension = "." + extension;
                RiskAttachmentDownload riskAttachmentDownload = new RiskAttachmentDownload
                {
                    Data = Convert.ToBase64String(fileResult.FileBytes),
                    ContentType = FileHelper.GetMimeTypeByExtension(extension),
                    FileName = riskAttachmentChunk.Url,
                    ChunkIndex = riskAttachmentChunk.ChunkIndex
                };
                return ServiceResponse<RiskAttachmentDownload>.ReturnResultWith200(riskAttachmentDownload);
            }
            return ServiceResponse<RiskAttachmentDownload>.ReturnFailed(500, fileResult.ErrorMessage);
        }
    }
}

