﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class MarkRiskAttachmentChunksUploadedCommandHandler (
        IRiskAttachmentRepository _riskAttachmentRepository,
        IRiskAttachmentChunkRepository _riskAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<MarkRiskAttachmentChunksUploadedCommandHandler> _logger
        ) : IRequestHandler<MarkRiskAttachmentChunksUploadedCommand, ServiceResponse<RiskAttachmentChunkStatus>>
    {
        public async Task<ServiceResponse<RiskAttachmentChunkStatus>> Handle(MarkRiskAttachmentChunksUploadedCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var riskAttachment = await _riskAttachmentRepository.All
                    .Where(c => c.Id == request.RiskAttachmentId).FirstOrDefaultAsync();

                if (riskAttachment == null)
                {
                    return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(404, "RiskAttachment not found.");
                }

                var riskChunks = _riskAttachmentChunkRepository
                    .All
                    .Where(c => c.RiskAttachmentId == riskAttachment.Id)
                    .OrderByDescending(c => c.ChunkIndex)
                    .ToList();

                if (riskChunks.Count == 0)
                {
                    return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(404, " RiskAttachment Chunks not found.");
                }

                if (riskAttachment.TotalChunk == riskChunks.Count())
                {
                    riskAttachment.IsAllChunkUploaded = request.Status;

                    _riskAttachmentRepository.Update(riskAttachment);

                    if (await _uow.SaveAsync(cancellationToken) < 0)
                    {
                        return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding riskAttachment");
                    }
                    var response = new RiskAttachmentChunkStatus { RiskAttachmentId = riskAttachment.Id, Status = request.Status };
                    return ServiceResponse<RiskAttachmentChunkStatus>.ReturnResultWith200(response);
                }
                else
                {
                    if (riskChunks.Count() > 0)
                    {
                        foreach (var documentChunk in riskChunks)
                        {
                            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(riskAttachment.StorageSettingId);
                            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                            await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                        }
                        _riskAttachmentChunkRepository.RemoveRange(riskChunks);
                    }

                    _riskAttachmentRepository.Remove(riskAttachment);
                    if (await _uow.SaveAsync() <= 0)
                    {
                        return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding riskAttachment");
                    }

                    return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
                }
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "Error while marking riskAttachment chunks uploaded.");
                return ServiceResponse<RiskAttachmentChunkStatus>.ReturnFailed(500, "Error while marking riskAttachment chunks uploaded.");
            }
        }
    }
}
