﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class UploadRiskAttachmentChunkCommandHandler(
        IRiskAttachmentRepository _riskAttachmentRepository,
        IRiskAttachmentChunkRepository _riskAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UploadRiskAttachmentChunkCommandHandler> _logger)
        : IRequestHandler<UploadRiskAttachmentChunkCommand, ServiceResponse<RiskAttachmentChunkDto>>
    {
        public async Task<ServiceResponse<RiskAttachmentChunkDto>> Handle(UploadRiskAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                if (request.File == null)
                {
                    return ServiceResponse<RiskAttachmentChunkDto>.ReturnFailed(409, "Please select the file.");
                }

                var existRiskttachment = _riskAttachmentRepository.Find(request.RiskAttachmentId);
                if (existRiskttachment == null)
                {
                    return ServiceResponse<RiskAttachmentChunkDto>.Return404("RiskAttachment not found");
                }
                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(existRiskttachment.StorageSettingId);

                var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);

                var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, existRiskttachment.Extension, existRiskttachment.Key, existRiskttachment.IV);

                if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
                {
                    return ServiceResponse<RiskAttachmentChunkDto>.Return422("Settings are not properly setup.");
                }
                var entity = new RiskAttachmentChunk()
                {
                    Id = Guid.NewGuid(),
                    ChunkIndex = request.ChunkIndex,
                    RiskAttachmentId = request.RiskAttachmentId,
                    Extension = request.Extension,
                    Size = request.Size,
                    Url = fileNameKeyValut.FileName
                };
                _riskAttachmentChunkRepository.Add(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskAttachmentChunkDto>.ReturnFailed(500, "Error While Added Document");
                }

                var entityDto = _mapper.Map<RiskAttachmentChunkDto>(entity);
                return ServiceResponse<RiskAttachmentChunkDto>.ReturnResultWith200(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while storing RiskAttachmentChunk");
                return ServiceResponse<RiskAttachmentChunkDto>.Return500("Error while storing RiskAttachmentChunk.");
            }
        }
    }
}
