﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Entities.RiskManagment;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class AddRiskControlCommandHandler(
        IRiskControlRepository _riskControlRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        IUserRepository _userRepository,
        IRiskRepository _riskRepository,
        ILogger<AddRiskControlCommandHandler> _logger,
        ISendEmailRepository _sendEmailRepository,
         UserInfoToken _userInfoToken,
         IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddRiskControlCommand, ServiceResponse<RiskControlDto>>
    {
        public async Task<ServiceResponse<RiskControlDto>> Handle(AddRiskControlCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entity = _mapper.Map<RiskControl>(request);
                entity.ReviewDate = request.ReviewDate == null ? DateTime.UtcNow : request.ReviewDate;
                if (request.Status == RiskStatusEnum.Closed || request.Status == RiskStatusEnum.Reviewed)
                {
                    var risk = await _riskRepository.All.Where(c => c.Id == entity.RiskId).FirstOrDefaultAsync();
                    var isPendingControl = await _riskControlRepository.All.Where(c => c.RiskId == entity.RiskId)
                       .AnyAsync(c => c.Status == RiskStatusEnum.Open ||
                                 c.Status == RiskStatusEnum.Monitoring ||
                                 c.Status == RiskStatusEnum.Escalated);
                    if (risk != null && !isPendingControl)
                    {
                        risk.Status = request.Status;
                        _riskRepository.Update(risk);
                    }
                }
                _riskControlRepository.Add(entity);
                // Send Email Responsible Person 
                try
                {
                    var userIds = new[] { _userInfoToken.Id, entity.ResponsibleId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var asignedUser = users.FirstOrDefault(u => u.Id == entity.ResponsibleId);
                    if (asignedUser != null)
                    {
                        var mainTitle = "Risk Control Assignment Notification";
                        var subject = "Risk Control Assignment";
                        var emailMessage = $"You have been assigned as the responsible person for a Risk Control : <b>{entity.Title}</b>. Please log in to the system to review the control details, implement the required control measures, and ensure it is completed and effective by the due date.";
                      _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = asignedUser.Email,
                            FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                            ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                            CreatedBy = fromUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailMessage, mainTitle, subject);

                        // Sent Notification
                        var notificationDto = new UserNotificationDto
                        {
                            Id = entity.Id,
                            UserId = entity.ResponsibleId,
                            NotificationsType = NotificationsType.RISK_CONTROL,
                            Title = entity.Title
                        };
                        _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskControlDto>.Return500();
                }
                var responsible = await _userRepository.All.Where(c => c.Id == entity.ResponsibleId).FirstOrDefaultAsync();
                if (responsible != null)
                {
                    entity.ResponsibleBy = responsible;
                }
                var entityDto = _mapper.Map<RiskControlDto>(entity);
                return ServiceResponse<RiskControlDto>.ReturnResultWith201(entityDto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving RiskControl");
                return ServiceResponse<RiskControlDto>.Return500("Error while saving RiskControl");
            }
        }
    }
}
