﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities.RiskManagment;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class UpdateRiskControlCommandHandler(
        IRiskControlRepository _riskControlRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        IUserRepository _userRepository,
        IRiskRepository _riskRepository,
        ILogger<AddRiskControlCommandHandler> _logger) : IRequestHandler<UpdateRiskControlCommand, ServiceResponse<RiskControlDto>>
    {
        public async Task<ServiceResponse<RiskControlDto>> Handle(UpdateRiskControlCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExist = await _riskControlRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (entityExist == null)
                {
                    return ServiceResponse<RiskControlDto>.Return404("risk Control not found");
                }
                _mapper.Map(request, entityExist);
                if (request.Status == RiskStatusEnum.Closed || request.Status == RiskStatusEnum.Reviewed)
                {
                    var risk = await _riskRepository.All.Where(c => c.Id == entityExist.RiskId).FirstOrDefaultAsync();
                    var isPendingControl = await _riskControlRepository.All.Where(c => c.RiskId == entityExist.RiskId && c.Id != request.Id)
                       .AnyAsync(c => c.Status == RiskStatusEnum.Open ||
                                 c.Status == RiskStatusEnum.Monitoring ||
                                 c.Status == RiskStatusEnum.Escalated);
                    if (risk != null && !isPendingControl)
                    {
                        risk.Status = request.Status;
                        _riskRepository.Update(risk);
                    }
                }
                _riskControlRepository.Update(entityExist);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskControlDto>.Return500();
                }
                var responsible = await _userRepository.All.Where(c => c.Id == entityExist.ResponsibleId).FirstOrDefaultAsync();
                if (responsible != null)
                {
                    entityExist.ResponsibleBy = responsible;
                }
                var entityDto = _mapper.Map<RiskControlDto>(entityExist);
                return ServiceResponse<RiskControlDto>.ReturnResultWith201(entityDto);
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "Error while updating RiskControl");
                return ServiceResponse<RiskControlDto>.Return500("Error while updating RiskControl");
            }
        }
    }
}
