﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR.RiskManagement
{
    public class UpdateRiskReviewCommandHandler(
        IRiskReviewRepository _riskReviewRepository,
        IRiskMatrixLevelRepository _riskMatrixLevelRepository,
        IMapper _mapper,
        ILogger<AddRiskReviewCommandHandler> _logger,
        IUnitOfWork<QMSDbContext> _uow,
        IUserRepository _userRepository,
        UserInfoToken _userInfoToken,
        ISendEmailRepository _sendEmailRepository,
        IUserNotificationRepository _userNotificationRepository,
        IRiskRepository _riskRepository) : IRequestHandler<UpdateRiskReviewCommand, ServiceResponse<RiskReviewDto>>
    {
        public async Task<ServiceResponse<RiskReviewDto>> Handle(UpdateRiskReviewCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExist = await _riskReviewRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (entityExist == null)
                {
                    return ServiceResponse<RiskReviewDto>.Return404("Risk Review not Found");
                }
                var risk = await _riskRepository.All.Where(c => c.Id == request.RiskId).FirstOrDefaultAsync();

                // If Reviewer not are Same Then sent Mail
                if (request.ReviewerId != entityExist.ReviewerId)
                {
                    try
                    {
                        var userIds = new[] { _userInfoToken.Id, request.ReviewerId };
                        var users = await _userRepository.All
                            .Where(u => userIds.Contains(u.Id))
                            .ToListAsync();
                         
                        var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                        var reviewerUser = users.FirstOrDefault(u => u.Id == request.ReviewerId);
                        if (reviewerUser != null)
                        {
                            var mainTitle = "Risk Review Assignment Notification";
                            var subject = "Risk Review Assignment";
                            var emailMessage = $"You have been assigned as the reviewer for a Risk : <b>{risk.Title}</b>. Please log in to the system to review the risk details, evaluate its likelihood and impact, assess the effectiveness of controls, and provide your comments and recommendations before the review date.";
                            _sendEmailRepository.AddAsignmentEmails(new SendEmail
                            {
                                Email = reviewerUser.Email,
                                FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                                ToName = reviewerUser.FirstName + ' ' + reviewerUser.LastName,
                                CreatedBy = fromUser.Id,
                                CreatedDate = DateTime.UtcNow,
                            }, emailMessage, mainTitle, subject);

                            // Sent Notification
                            var notificationDto = new UserNotificationDto
                            {
                                Id = entityExist.Id,
                                UserId = request.ReviewerId.Value,
                                NotificationsType = NotificationsType.RISK_REVIEW,
                                Title = risk != null ? risk.Title : ""
                            };
                            _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                        }
                    }
                    catch (System.Exception ex)
                    {
                        _logger.LogError(ex, "email does not sent");
                    }

                }
                _mapper.Map(request, entityExist);
                int riskScore = request.Likelihood * request.Impact;
                var riskLevel = await _riskMatrixLevelRepository
                    .All
                    .Where(c => c.ScoreMin <= riskScore && c.ScoreMax >= riskScore).FirstOrDefaultAsync();
                entityExist.RiskScore = riskScore;
                entityExist.ReviewerId = request.ReviewerId == null ? _userInfoToken.Id : request.ReviewerId.Value;
                entityExist.RiskLevel = riskLevel.RiskLevel;
                entityExist.ReviewDate = request.ReviewDate == DateTime.MinValue ? DateTime.UtcNow : request.ReviewDate;
                _riskReviewRepository.Update(entityExist);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<RiskReviewDto>.Return500();
                }
                var reviewer = await _userRepository.All.Where(c => c.Id == entityExist.ReviewerId).FirstOrDefaultAsync();
                if (reviewer != null)
                {
                    entityExist.Reviewer = reviewer;
                }
                var entityDto = _mapper.Map<RiskReviewDto>(entityExist);
                return ServiceResponse<RiskReviewDto>.ReturnResultWith201(entityDto);
            }

            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Erorr while saving Riskreview");
                return ServiceResponse<RiskReviewDto>.Return500("Error while saving RiskReview");
            }
        }
    }
}
